
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2007 by Wolfgang Hoffmann <woho@woho.de>                *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/


#include "settings.h"
#include "ui_settings.h"
#include "log.h"

#include <QtGui>




/********************************************************************/
/*!
    \class Settings
    \brief Container for application settings

*/

Settings Settings::g_oSettings;


Settings::Settings()
:   m_bFrozen(false),
    m_bChanged(false)
    {
    QCoreApplication::setOrganizationName("woho");
    QCoreApplication::setOrganizationDomain("woho.de");
    QCoreApplication::setApplicationName("aspect");
    m_pQSettings = new QSettings(QSettings::IniFormat, QSettings::UserScope,
        QString::fromUtf8("woho"), QString::fromUtf8("aspect"));
    }

Settings::~Settings()
    {
    delete m_pQSettings;
    }

void Settings::emitChanged()
    {
    g_oSettings.emitChangedInternal(true);
    }

void Settings::connect(QObject *pReceiver, const char *szSlot)
    {
    QObject::connect(&g_oSettings, SIGNAL(settingsChanged()), pReceiver, szSlot);
    }

void Settings::freeze()
    {
    g_oSettings.m_bFrozen = true;
    }

void Settings::thaw()
    {
    g_oSettings.m_bFrozen = false;
    g_oSettings.emitChangedInternal(false);
    }

void Settings::emitChangedInternal(bool bFromChange)
    {
    if (bFromChange)
        m_bChanged = true;
    if (m_bChanged && (!m_bFrozen))
        {
        m_bChanged = false;
        emit settingsChanged();
        }
    }


QString Settings::getPathLog()
    {
    return QDir::toNativeSeparators(settings()
        .value("path/log", QDir::tempPath()).toString());
    }

void Settings::setPathLog(const QString &qsPath)
    {
    if (qsPath == getPathLog())
        return;
    QString qsPathSep = QDir::fromNativeSeparators(qsPath);
    Settings::settings().setValue("path/log", qsPathSep);
    emitChanged();
    }


QStringList Settings::getPathHistory()
    {
    QStringList qslHistorySep = settings()
        .value("path/history", QStringList()).toStringList();
    QStringList qslHistory;
    for (int nCnt = 0; nCnt < qslHistorySep.size(); nCnt++)
        qslHistory.append(QDir::toNativeSeparators(qslHistorySep.at(nCnt)));
    return qslHistory;
    }

QString Settings::getPathDefault()
    {
    return QDir::toNativeSeparators(settings()
        .value("path/current", QDir::home().absolutePath()).toString());
    }

void Settings::setPathDefault(const QString &qsPath)
    {
    if (qsPath == getPathDefault())
        return;
    QString qsPathSep = QDir::fromNativeSeparators(qsPath);
    Settings::settings().setValue("path/current", qsPathSep);
    // add current path to end of history
    // remove any duplicates and keep list below 20 entries
    QStringList qslHistory = Settings::getPathHistory();
    for (int nCnt = qslHistory.size() - 1; nCnt >= 0; nCnt--)
        if (qslHistory.at(nCnt) == qsPathSep)
            qslHistory.removeAt(nCnt);
    qslHistory.append(qsPathSep);
    while (qslHistory.size() > 20)
        qslHistory.removeFirst();
    Settings::settings().setValue("path/history", qslHistory);
    emitChanged();
    }


bool Settings::getLoggingGeneral()
    {
    return settings().value("log/general", true).toBool();
    }

void Settings::setLoggingGeneral(bool bOn)
    {
    if (bOn == getLoggingGeneral())
        return;
    Settings::settings().setValue("log/general", bOn);
    emitChanged();
    }

bool Settings::getLoggingMcu()
    {
    return settings().value("log/mcu", false).toBool();
    }

void Settings::setLoggingMcu(bool bOn)
    {
    if (bOn == getLoggingMcu())
        return;
    Settings::settings().setValue("log/mcu", bOn);
    emitChanged();
    }


QString Settings::getMcuPort()
    {
    return settings().value("mcu/serialport", QString::fromUtf8("COM1")).toString();
    }

void Settings::setMcuPort(const QString &qsPort)
    {
    if (qsPort == getMcuPort())
        return;
    Settings::settings().setValue("mcu/serialport", qsPort);
    emitChanged();
    }

int Settings::getMcuFirmware()
    {
    return settings().value("mcu/firmware", 35904).toInt() / 100;
    }

void Settings::setMcuFirmware(int nFirmware)
    {
    // store as ((major*100)+minor)*100+revision
    // revision is currently not considered
    int nCurrent = getMcuFirmware();
    if (nFirmware < 100)
        {
        nFirmware *= 100;
        nCurrent = (nCurrent / 100) * 100;
        }
    if (nFirmware == nCurrent)
        return;
    Settings::settings().setValue("mcu/firmware", nFirmware * 100);
    emitChanged();
    }

int Settings::getMcuWormWheelTeeth()
    {
    return settings().value("mcu/wormwheelteeth", 180).toInt();
    }

void Settings::setMcuWormWheelTeeth(int nTeeth)
    {
    if (nTeeth == getMcuWormWheelTeeth())
        return;
    Settings::settings().setValue("mcu/wormwheelteeth", nTeeth);
    emitChanged();
    }


int Settings::getPecGenTraversal()
    {
    return settings().value("pecGen/traversal", -1).toInt();
    }

void Settings::setPecGenTraversal(int nTraversal)
    {
    if (nTraversal == getPecGenTraversal())
        return;
    Settings::settings().setValue("pecGen/traversal", nTraversal);
    emitChanged();
    }

double Settings::getPecGenAccel()
    {
    return settings().value("pecGen/accel", 240.).toDouble();
    }

void Settings::setPecGenAccel(double dfAccel)
    {
    if (dfAccel == getPecGenAccel())
        return;
    Settings::settings().setValue("pecGen/accel", dfAccel);
    emitChanged();
    }

int Settings::getPecGenAggressMin()
    {
    return settings().value("pecGen/aggressMin", 1).toInt();
    }

void Settings::setPecGenAggressMin(int nAggressMin)
    {
    if (nAggressMin == getPecGenAggressMin())
        return;
    Settings::settings().setValue("pecGen/aggressMin", nAggressMin);
    emitChanged();
    }

int Settings::getPecGenAggressMax()
    {
    return settings().value("pecGen/aggressMax", 15).toInt();
    }

void Settings::setPecGenAggressMax(int nAggressMax)
    {
    if (nAggressMax == getPecGenAggressMax())
        return;
    Settings::settings().setValue("pecGen/aggressMax", nAggressMax);
    emitChanged();
    }

int Settings::getPecGenFactorMax()
    {
    return settings().value("pecGen/factorMax", 15).toInt();
    }

void Settings::setPecGenFactorMax(int nFactorMax)
    {
    if (nFactorMax == getPecGenFactorMax())
        return;
    Settings::settings().setValue("pecGen/factorMax", nFactorMax);
    emitChanged();
    }




/********************************************************************/
/*!
    \class SettingsDialog
    \brief GUI for application settings

    \todo implement logfile path selection
    \todo implement settings export/import
*/

SettingsDialog::SettingsDialog(QWidget *pParent)
:   QDialog(pParent)
    {
    setAttribute(Qt::WA_DeleteOnClose, false);
    setAttribute(Qt::WA_QuitOnClose, false);

    m_pUi = new Ui::SettingsDialog;
    m_pUi->setupUi(this);

    m_pUi->logfilesPath->setText(
        Settings::settings().value("path/logfiles", QDir::tempPath()).toString());

    m_pUi->pecGenTraversal->setCurrentIndex((Settings::getPecGenTraversal() > 0)? 1: 0);
    m_pUi->pecGenAccel->setValue(Settings::getPecGenAccel());
    m_pUi->pecGenAggressMin->setValue(Settings::getPecGenAggressMin());
    m_pUi->pecGenAggressMax->setValue(Settings::getPecGenAggressMax());
    m_pUi->pecGenFactor->setValue(Settings::getPecGenFactorMax());

    // todo: implement functionality
    m_pUi->logfilesPathSelect->hide();
    m_pUi->settingsImport->hide();
    m_pUi->settingsExport->hide();
    }


SettingsDialog::~SettingsDialog()
    {
    delete m_pUi;
    }


void SettingsDialog::on_logfilesGeneral_clicked()
    {
    Settings::setLoggingGeneral(m_pUi->logfilesGeneral->isChecked());
    }


void SettingsDialog::on_logfilesMcu_clicked()
    {
    Settings::setLoggingMcu(m_pUi->logfilesMcu->isChecked());
    }


void SettingsDialog::on_logfilesPathSelect_clicked()
    {
    LOG(4, "SettingsDialog::on_logfilesPathSelect_clicked");
    /* todo: open file requester and ask ...
    Settings::settings().setValue("path/logfiles", QDir::tempPath());
    */
    }


void SettingsDialog::on_pecGenTraversal_activated(int nIndex)
    {
    Settings::setPecGenTraversal((nIndex == 0)? -1: 1);
    }


void SettingsDialog::on_pecGenAccel_valueChanged(double dfValue)
    {
    Settings::setPecGenAccel(dfValue);
    }


void SettingsDialog::on_pecGenAggressMin_valueChanged(int nValue)
    {
    Settings::setPecGenAggressMin(nValue);
    }


void SettingsDialog::on_pecGenAggressMax_valueChanged(int nValue)
    {
    Settings::setPecGenAggressMax(nValue);
    }


void SettingsDialog::on_pecGenFactor_valueChanged(int nValue)
    {
    Settings::setPecGenFactorMax(nValue);
    }


