
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2007 by Wolfgang Hoffmann <woho@woho.de>                *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/


#include "mount.h"
#include "ui_mount.h"
#include "data.h"
#include "log.h"
#include "mcuproxy.h"
#include "settings.h"

#include <QtGui>
#include <math.h>




/********************************************************************/
/*!
    \class MountDialog
    \brief Experimental GUI for configuring MCU
*/

MountDialog::MountDialog(QWidget *pParent)
:   QDialog(pParent)
    {
    setAttribute(Qt::WA_DeleteOnClose, false);
    setAttribute(Qt::WA_QuitOnClose, false);

    m_pUi = new Ui::MountDialog;
    m_pUi->setupUi(this);

    // Insert edited item into list when combobox looses focus.
    // Qt should do this ...
    if (m_pUi->mcuPort->lineEdit() != 0)
        connect(m_pUi->mcuPort->lineEdit(), SIGNAL(editingFinished()),
            m_pUi->mcuPort->lineEdit(), SIGNAL(returnPressed()));

    Settings::connect(this, SLOT(settingsChanged()));
    settingsChanged();

    m_pMcuProxy = &(McuProxy::proxy());
    connect(m_pMcuProxy, SIGNAL(busy(bool)),
        this, SLOT(mcuSerialBusy(bool)));
    }


MountDialog::~MountDialog()
    {
    delete m_pUi;
    }


void MountDialog::settingsChanged()
    {
    QString qsPort = Settings::getMcuPort();
    int nIndex = m_pUi->mcuPort->findText(qsPort);
    if (nIndex == -1)
        {
        nIndex = m_pUi->mcuPort->count();
        m_pUi->mcuPort->insertItem(nIndex, qsPort);
        }
    m_pUi->mcuPort->setCurrentIndex(nIndex);
    }


void MountDialog::mcuSerialBusy(bool bBusy)
    {
    if (bBusy)
        {
        m_pUi->mcuFirmwareRefresh->setEnabled(false);
        m_pUi->mcuSettingsTracking->setEnabled(false);
        m_pUi->mcuSettingsPec->setEnabled(false);
        m_pUi->mcuSettingsIntelly->setEnabled(false);
        m_pUi->mcuSettingsRefresh->setEnabled(false);
        m_pUi->mcuGearsQuery->setEnabled(false);
        m_pUi->mcuGearsSet->setEnabled(false);
        m_pUi->mcuCommandSend->setEnabled(false);
        }
    else
        {
        m_pUi->mcuFirmwareRefresh->setEnabled(true);
        m_pUi->mcuSettingsTracking->setEnabled(true);
        m_pUi->mcuSettingsPec->setEnabled(true);
        m_pUi->mcuSettingsIntelly->setEnabled(true);
        m_pUi->mcuSettingsRefresh->setEnabled(true);
        m_pUi->mcuGearsQuery->setEnabled(true);
        m_pUi->mcuGearsSet->setEnabled(true);
        m_pUi->mcuCommandSend->setEnabled(true);
        }
    }




/********************************************************************/
/*
    MCU connection
*/


void MountDialog::on_mcuPort_activated(const QString &qsPort)
    {
    Settings::setMcuPort(qsPort);
    }


void MountDialog::on_mcuFirmwareRefresh_clicked()
    {
    McuDataFirmware oFirmware;
    McuDataKeys oKeys;
    if (mcuCommunicationBegin())
        {
        oFirmware = m_pMcuProxy->getFirmware();
        oKeys = m_pMcuProxy->getKeys();
        mcuCommunicationEnd();
        }

    m_pUi->mcuFirmwareVersion->setText(oFirmware.firmware());
    m_pUi->mcuFirmwareSerial->setText(QString::fromUtf8("%1")
        .arg(oKeys.serial()));
    m_pUi->mcuFirmwareKeys->setText(QString::fromUtf8("Goto: %1, IntellyTrack®: %2")
        .arg(QString::fromUtf8(oKeys.keyGoto()? "yes": "no"))
        .arg(QString::fromUtf8(oKeys.keyIntelly()? "yes": "no")));
    }


void MountDialog::on_mcuSettingsTracking_activated(int nSpeed)
    {
    if (mcuCommunicationBegin())
        {
        m_pMcuProxy->setTrackingRate(
            (nSpeed == 0)? McuProxy::sidereal:
            (nSpeed == 1)? McuProxy::solar:
            (nSpeed == 2)? McuProxy::lunar:
            McuProxy::off);
        mcuCommunicationEnd();
        }
    }


void MountDialog::on_mcuSettingsPec_activated(int nIndex)
    {
    if (m_pUi->mcuSettingsPec->count() == 1)
        return;
    if (mcuCommunicationBegin())
        {
        m_pMcuProxy->setPec(nIndex == 1);
        mcuCommunicationEnd();
        }
    }


void MountDialog::on_mcuSettingsIntelly_activated(int nIndex)
    {
    if (mcuCommunicationBegin())
        {
        m_pMcuProxy->setIntellyTrack(nIndex == 1);
        mcuCommunicationEnd();
        }
    }


void MountDialog::on_mcuSettingsRefresh_clicked()
    {
    int nTrackingRate = -1;
    //McuDataState oState;
    McuDataPecState oPec;
    bool bIntellyTrack = false;
    QString qsRotOrient;
    if (mcuCommunicationBegin())
        {
        switch (m_pMcuProxy->getTrackingRate())
            {
            case McuProxy::sidereal: nTrackingRate = 0; break;
            case McuProxy::solar:    nTrackingRate = 1; break;
            case McuProxy::lunar:    nTrackingRate = 2; break;
            case McuProxy::off:      nTrackingRate = 3; break;
            }
        oPec = m_pMcuProxy->getPecState();
        bIntellyTrack = m_pMcuProxy->getIntellyTrack();
        //qsRotOrient = m_pMcuProxy->getRotOrient();
        mcuCommunicationEnd();
        }

    m_pUi->mcuSettingsTracking->setCurrentIndex(nTrackingRate);

    // dynamically change contents of PEC dropdown, as single entries cannot be disabled
    if (oPec.pec() < 2)
        {
        m_pUi->mcuSettingsPec->setItemText(0, tr("Off"));
        if (m_pUi->mcuSettingsPec->count() == 1)
            m_pUi->mcuSettingsPec->addItem(tr("On"));
        else
            m_pUi->mcuSettingsPec->setItemText(1, tr("On"));
        m_pUi->mcuSettingsPec->setCurrentIndex(oPec.pec());
        }
    else
        {
        if (m_pUi->mcuSettingsPec->count() == 2)
            m_pUi->mcuSettingsPec->removeItem(1);
        if (oPec.pec() == 2)
            m_pUi->mcuSettingsPec->setItemText(0, tr("On by Handbox"));
        else
            m_pUi->mcuSettingsPec->setItemText(0, tr("In Training"));
        }

    m_pUi->mcuSettingsIntelly->setCurrentIndex(bIntellyTrack? 1: 0);

    /* todo
    m_pUi->mcuSettingsRates->setText(
        QString::fromUtf8("2x: %1, 8x: %2, 16x: %3, Goto: %4, Ramp: %5")
        .arg(oState.guideRate())
        .arg(oState.paddle8Rate())
        .arg(oState.paddle16Rate())
        .arg(oState.gotoRate())
        .arg(oState.rampSteepness()));
    m_pUi->mcuSettingsMount->setText(qsRotOrient);
    m_pUi->mcuSettingsBacklash->setText(oState.backlash());
    */
    }


void MountDialog::on_mcuGearsQuery_clicked()
    {
    int nRotation = 0;
    int nOrientation = 0;
    McuDataGears oGears;
    if (mcuCommunicationBegin())
        {
        switch (m_pMcuProxy->getRotation())
            {
            case McuProxy::original: nRotation = 0; break;
            case McuProxy::reverse:  nRotation = 1; break;
            }
        switch (m_pMcuProxy->getOrientation())
            {
            case McuProxy::left:  nOrientation = 0; break;
            case McuProxy::right: nOrientation = 1; break;
            }
        oGears = m_pMcuProxy->getGears();
        mcuCommunicationEnd();
        }

    m_pUi->mcuGearsRaWorm->setValue(oGears.raWormWheel());
    m_pUi->mcuGearsRaGear->setValue(oGears.raGearRatio());
    m_pUi->mcuGearsRaStep->setValue(oGears.raMotorStepAngle());
    m_pUi->mcuGearsDeWorm->setValue(oGears.deWormWheel());
    m_pUi->mcuGearsDeGear->setValue(oGears.deGearRatio());
    m_pUi->mcuGearsDeStep->setValue(oGears.deMotorStepAngle());

    McuDataGears oCheck(
        oGears.raWormWheel(), oGears.raGearRatio(), oGears.raMotorStepAngle(),
        oGears.deWormWheel(), oGears.deGearRatio(), oGears.deMotorStepAngle());
    m_pUi->mcuGearsState->setText(
        QString::fromUtf8((oGears == oCheck)? "ok": "inconsistent"));
    m_pUi->mcuGearsInternalClock->setText(
        QString::fromUtf8("sid: %1,%2 sol: %3,%4 lun: %5,%6")
        .arg(oGears.clockDivSid())
        .arg(oGears.clockAdjSid())
        .arg(oGears.clockDivSol())
        .arg(oGears.clockAdjSol())
        .arg(oGears.clockDivLun())
        .arg(oGears.clockAdjLun()));
    m_pUi->mcuGearsInternalCount->setText(
        QString::fromUtf8("goto %7,%8, pec %9")
        .arg(oGears.countGotoRa())
        .arg(oGears.countGotoDe())
        .arg(oGears.countPec()));

    m_pUi->mcuMountRotation->setCurrentIndex(nRotation);
    m_pUi->mcuMountOrientation->setCurrentIndex(nOrientation);
    }


void MountDialog::on_mcuGearsSet_clicked()
    {
    McuDataGears oGears(
        m_pUi->mcuGearsRaWorm->value(),
        m_pUi->mcuGearsRaGear->value(),
        m_pUi->mcuGearsRaStep->value(),
        m_pUi->mcuGearsDeWorm->value(),
        m_pUi->mcuGearsDeGear->value(),
        m_pUi->mcuGearsDeStep->value());
    bool bConsistent = false;

    if (mcuCommunicationBegin())
        {
        m_pMcuProxy->setRotation(
            (m_pUi->mcuMountRotation->currentIndex() == 0)? McuProxy::original: McuProxy::reverse);
        m_pMcuProxy->setOrientation(
            (m_pUi->mcuMountRotation->currentIndex() == 0)? McuProxy::left: McuProxy::right);
        bConsistent = m_pMcuProxy->setGears(oGears);
        mcuCommunicationEnd();
        }

    m_pUi->mcuGearsState->setText(
        QString::fromUtf8(bConsistent? "ok": "inconsistent"));
    m_pUi->mcuGearsInternalClock->setText(
        QString::fromUtf8("sid: %1,%2 sol: %3,%4 lun: %5,%6")
        .arg(oGears.clockDivSid())
        .arg(oGears.clockAdjSid())
        .arg(oGears.clockDivSol())
        .arg(oGears.clockAdjSol())
        .arg(oGears.clockDivLun())
        .arg(oGears.clockAdjLun()));
    m_pUi->mcuGearsInternalCount->setText(
        QString::fromUtf8("goto %7,%8, pec %9")
        .arg(oGears.countGotoRa())
        .arg(oGears.countGotoDe())
        .arg(oGears.countPec()));
    }


void MountDialog::on_mcuCommandSend_clicked()
    {
    LOG(4, "MountDialog::on_mcuCommandSend_clicked");
    QByteArray aCommand = m_pUi->mcuCommand->currentText().toLatin1();
    QByteArray aAnswer;
    QApplication::setOverrideCursor(Qt::BusyCursor);
    if (m_pMcuProxy->open(m_pUi->mcuPort->currentText(), true))
        {
        aAnswer = m_pMcuProxy->mcuRawCommand(aCommand);
        m_pMcuProxy->close();
        }
    QApplication::restoreOverrideCursor();
    QString qsAnswer;
    for (int nByte = 0; nByte < aAnswer.length(); nByte++)
        qsAnswer += QString::fromUtf8("%1 ")
            .arg((int)((unsigned char)aAnswer.at(nByte)), 2, 16, QLatin1Char('0'));
    qsAnswer += '"';
    for (int nByte = 0; nByte < aAnswer.length(); nByte++)
        {
        QChar oChar = QChar::fromLatin1(aAnswer.at(nByte));
        qsAnswer += (oChar.isPrint())? oChar: QLatin1Char('.');
        }
    qsAnswer += '"';
    m_pUi->mcuAnswer->setText(qsAnswer);
    }


bool MountDialog::mcuCommunicationBegin()
    {
    QApplication::setOverrideCursor(Qt::BusyCursor);
    bool bSuccess = m_pMcuProxy->open(m_pUi->mcuPort->currentText());
    QApplication::restoreOverrideCursor();
    return bSuccess;
    }


void MountDialog::mcuCommunicationEnd()
    {
    m_pMcuProxy->close();
    }


