
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2006-2007 by Wolfgang Hoffmann <woho@woho.de>           *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/


#include "mcuserial.h"
#include "log.h"

#include "qextserialport.h"

#include <QtCore>




/********************************************************************/
/*!
    \class IMcuCmd
    \brief Interface for MCU serial communication command

    This class is, together with IMcuAns, an approach to generically
    deal with the variety of different command and answer types.

    \sa IMcuCmd
*/


/*!
    \fn QByteArray IMcuCmd::command() const
    Get raw command string for submission to the MCU.
*/


/*!
    \fn QString IMcuCmd::name() const
    Return short human-readable name of command for debugging.
*/




/********************************************************************/
/*!
    \class IMcuAns
    \brief Interface for MCU serial communication answer

    This class is, together with IMcuCmd, an approach to generically
    deal with the variety of different command and answer types.

    \sa IMcuCmd
*/


/*!
    \fn void receive(void McuAns::receive(int &rnMin, int &rnMax,
        const QByteArray &aAnswerFragment)

    McuSerial uses this method to hand over raw characters
    received from the MCU in \a aAnswerFragment.

    The return values \a rnMin and \a rnMax must indicate how
    much more answer characters are expected. See McuSerial::recv()
    for detailed meaning.

    \a rnMin and \a rnMax can change between subsequent calls of
    receive(), since commands may return a status character and,
    if status is not ok, additional descriptive characters.
*/


/*!
    \fn QString IMcuAns::name() const
    Return short human-readable name of command for debugging.
*/




/********************************************************************/
/*!
    \class Sleeper
    \brief Helper class for sleeping a bit
    \internal
*/

class Sleeper: public QThread
    {
public:
    static void msleep(unsigned long nMSecs)
        { QThread::msleep(nMSecs); };
    };




/********************************************************************/
/*!
    \class McuSerial
    \brief MCU serial communication handler

    This class encapsulates the serial port that is used to
    talk to the MCU.

    It provides methods for passing commands described by IMcuCmd
    derived classes to the MCU and filling in the received answers.
*/


McuSerial::McuSerial()
:   m_pSerPort(0),
    m_bOpen(false),
    m_eStatus(unknown)
    {
    }


McuSerial::~McuSerial()
    {
    close();
    }


bool McuSerial::open(const QString &qsPort)
    {
    close();

    QString qsPortName = qsPort;
#ifdef Q_WS_WIN
    // "COM1", "COM2", ...
#else
    // "/dev/ttyS0", "/dev/ttyS1", ...
    if (qsPortName.startsWith(QString::fromUtf8("COM")))
        {
        int nPort = qsPortName.remove(0, 3).toInt();
        if (nPort > 0)
            qsPortName = QString::fromUtf8("/dev/ttyS%1").arg(nPort - 1);
        }
#endif

    LOG_MCU(QString::fromUtf8("trying to open serial port \"%1\" ...\n").arg(qsPortName));
    m_pSerPort = new QextSerialPort(qsPortName);
    m_pSerPort->setBaudRate(QextSerialPort::BAUD9600);
    m_pSerPort->setDataBits(QextSerialPort::DATA_8);
    m_pSerPort->setParity(QextSerialPort::PAR_NONE);
    m_pSerPort->setStopBits(QextSerialPort::STOP_1);
    m_pSerPort->setFlowControl(QextSerialPort::FLOW_OFF);

    m_bOpen = m_pSerPort->open();
    if (!m_bOpen)
        {
        m_eStatus = error;
        LOG(3, "McuSerial::open(%s): failed to open port \"%s\"", LQS(qsPort), LQS(qsPortName));
        LOG_MCU(QString::fromUtf8("failed to open serial port \"%1\".\n").arg(qsPortName));
        emit status(m_bOpen, m_eStatus);
        return false;
        }

    m_eStatus = ans_ok;
    LOG_MCU(QString::fromUtf8("successfully opened serial port \"%1\" ...\n").arg(qsPortName));
    emit status(m_bOpen, m_eStatus);
    return true;
    }


void McuSerial::close()
    {
    if (m_bOpen)
        {
        LOG_MCU(QString::fromUtf8("closing serial port \"%1\"\n").arg(m_pSerPort->name()));
        m_pSerPort->close();
        }
    m_bOpen = false;
    emit status(m_bOpen, m_eStatus);

    delete m_pSerPort;
    m_pSerPort = 0;
    }


/*!
    Send command \a pCmd over serial line to MCU

    Typically send() is followed by an immediate recv(). Except a few
    special cases, a flush() should be issued before send().

    \note: Care must be taken with conversions from/to QChar and QString:
    - Strings that are sent/received from the MCU must be converted
      with toLatin1()/fromLatin1()
    - Strings that come from source code must be converted with
      the editor's encoding, here fromUtf8()
    - Strings that are output to the debugging console must be
      converted with LQS() (i.e. toLocal8Bit())

    \sa recv(), flush()
*/

bool McuSerial::send(IMcuCmd *pCmd)
    {
    if (pCmd == 0)
        return false;

    if (!m_bOpen)
        {
        m_eStatus = error;
        LOG(3, "McuSerial::send(%s): port not open", LQS(pCmd->name()));
        LOG_MCU(QString::fromUtf8("cmd \"%1\" failed: serial port \"%2\" not open\n")
            .arg(pCmd->name()).arg((m_pSerPort == 0)? QString::fromUtf8("???"): m_pSerPort->name()));
        emit status(m_bOpen, m_eStatus);
        QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);
        return false;
        }

    QByteArray aCmd = pCmd->command();
    m_qsLastCmd = pCmd->name();

    // debugging
    {
    QString qsCmdHex = hex(aCmd);
    LOG(4, "send: %s", LQS(qsCmdHex));
    LOG_MCU(QString::fromUtf8("cmd \"%1\": sending ... %2\n")
        .arg(pCmd->name()).arg(qsCmdHex));
    }

    // send command
    if (-1 == m_pSerPort->write(aCmd))
        {
        m_eStatus = error;
        LOG(3, "McuSerial::send(%s): failed to send (port \"%s\")",
            LQS(pCmd->name()), LQS(m_pSerPort->name()));
        LOG_MCU(QString::fromUtf8("cmd \"%1\": failed to send (port \"%2\")\n")
            .arg(pCmd->name()).arg(m_pSerPort->name()));
        emit status(m_bOpen, m_eStatus);
        QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);
        return false;
        }

    // note: QextSerPort doesn't implement waitForBytesWritten()

    m_eStatus = cmd_ok;
    emit status(m_bOpen, m_eStatus);
    QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);
    return true;
    }

/*!
    Recieve answer \a pAns from MCU serial line

    Typically recv() is called immediately after send().

    Since the communication with QExtSerialPort seems to be very fragile
    when attempting to read more answer characters than available, an
    iterative approach is chosen:
    - query \a pAns for the expected answer length with IMcuAns::receive()
    - read all currently available answer characters up to rnMax
        from the serial line; if less than rnMin bytes are available,
        wait 50 msecs and retry receiving one time.
    - hand over received data to \a pAns, again with IMcuAns::receive().
        If further answer data is expected (returned number != 0), loop.
        If looping 4 times with rnMin > 0 and no answer data available,
        abort with a timeout log message. This is important to not block
        the GUI if e.g. connection to the MCU broke down.

    \note: Care must be taken with conversions from/to QChar and QString:
    - Strings that are sent/received from the MCU must be converted
      with toLatin1()/fromLatin1()
    - Strings that come from source code must be converted with
      the editor's encoding, here fromUtf8()
    - Strings that are output to the debugging console must be
      converted with LQS() (i.e. toLocal8Bit())

    \sa send()
*/

bool McuSerial::recv(IMcuAns *pAns, int nTimeout)
    {
    if (pAns == 0)
        return false;

    if (!m_bOpen)
        {
        m_eStatus = error;
        LOG(3, "McuSerial::recv(%s): port not open", LQS(pAns->name()));
        LOG_MCU(QString::fromUtf8("ans \"%1\" failed: serial port \"%2\" not open\n")
            .arg(pAns->name()).arg((m_pSerPort == 0)? QString::fromUtf8("???"): m_pSerPort->name()));
        emit status(m_bOpen, m_eStatus);
        QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);
        return false;
        }
    LOG_MCU(QString::fromUtf8("ans \"%1\": receiving ... ")
        .arg(pAns->name()));

    // receive anwer
    QByteArray aFullAns;
    QString qsError = QString();
    int nAnsMin, nAnsMax;
    bool bSuccess = true;
    for (pAns->receive(nAnsMin, nAnsMax, QByteArray()); nAnsMax > 0; )
        {
        // query how much characters are available at the serial port.
        // if less than the expected, wait 10 ms and query again.
        // bail out after a timeout to avoid spinning and thus blocking the GUI
        // if not enough data comes in (when e.g. connection broke down)
        int nBytesAvail = m_pSerPort->bytesAvailable();
        for (int nRetry = 0; nRetry < (nTimeout - 1) / 50 + 1; nRetry++)
            {
            if (nBytesAvail >= nAnsMin)
                break;
            LOG(6, "McuSerial::recv(%s): wait a bit; %d of %d bytes avail (port \"%s\")",
                LQS(pAns->name()), nBytesAvail, nAnsMin, LQS(m_pSerPort->name()));
            LOG_MCU(QString::fromUtf8("."));
            // sleep 50 ms here; don't go lower, as e.g. 10 ms might
            // be too short for time granularity on some systems ...
            msleep(50);
            QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents, 50);
            nBytesAvail = m_pSerPort->bytesAvailable();
            }
        // if no data arrived, again be careful to avoid spinning:
        // quit receive loop and set success state according to expected answer
        if (nBytesAvail == 0)
            {
            if (nAnsMin > 0)
                {
                LOG(3, "McuSerial::recv(%s): timeout; %d more bytes expected (port \"%s\")",
                    LQS(pAns->name()), nAnsMin, LQS(m_pSerPort->name()));
                qsError = QString::fromUtf8(
                    " timeout at receive; %1 more bytes expected (port \"%2\")")
                    .arg(nAnsMin).arg(m_pSerPort->name());
                bSuccess = false;
                }
            break;
            }
        // receive available characters, but not more than nAnsMax
        int nAnsLen = nAnsMax;
        if (nAnsLen > nBytesAvail)
            nAnsLen = nBytesAvail;
        QByteArray aAns = m_pSerPort->read(nAnsLen);
        if (aAns.size() < nAnsLen)
            {
            LOG(3, "McuSerial::recv(%s): failed to receive (port \"%s\")",
                LQS(pAns->name()), LQS(m_pSerPort->name()));
            LOG_MCU(QString::fromUtf8("!"));
            msleep(50);
            }
        aFullAns += aAns;
        pAns->receive(nAnsMin, nAnsMax, aAns);
        }
    m_eStatus = (bSuccess)? ans_ok: error;

    // debugging
    {
    QString qsAnsHex = hex(aFullAns);
    LOG(4, "recv: %s", LQS(qsAnsHex));
    LOG_MCU(QString::fromUtf8("%1%2\n").arg(qsAnsHex).arg(qsError));
    }

    emit status(m_bOpen, m_eStatus);
    QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);
    return bSuccess;
    }


/*!
    Flush serial receive queue.

    For most commands, it is recommended to flush the receive queue
    befor sending a new command, to avoid stale answer bytes spoiling
    the real answer to this command.

    There are exceptions where flushing is not wanted, i.e. while
    receiving a PEC table, where commands for reading each PEC value
    are issued and then all answers are read (for performance improvement).

    \sa send(), recv()
*/

void McuSerial::flush()
    {
    // read any orphaned answer bytes
    if (m_pSerPort->bytesAvailable() > 0)
        {
        // orphaned answer bytes are available; flush them
        m_eStatus = cmd_ok;
        int nRemain = m_pSerPort->bytesAvailable();
        LOG(3, "McuSerial::flush(%s): discarding %d orphaned answer bytes",
            LQS(m_qsLastCmd), nRemain);
        LOG_MCU(QString::fromUtf8("flush(%1): discarding %2 orphaned answer bytes:")
            .arg(m_qsLastCmd).arg(nRemain));
        QByteArray aAns = m_pSerPort->readAll();
        QString qsAnsHex = hex(aAns);
        LOG(3, "flush: %s", LQS(qsAnsHex));
        LOG_MCU(QString::fromUtf8(" %1\n").arg(qsAnsHex));
        emit status(m_bOpen, m_eStatus);
        }
    }


/*!
    \fn void log(const QString &qsText)

    Signal emitted when a string \a qsText for logging output
    is available.

    Connecting this signal might affect performance.
    To keep performance impact low when no logging is needed,
    the logging string is only constructed if at least one slot
    is connected to this signal.
*/


/*!
    Return binary data \a oData as hexdump and string of
    characters, with non-printable characters replaced by "."
*/

QString McuSerial::hex(const QByteArray &aData)
    {
    int nNum = aData.size();
    QString qsData;
    for (int nByte = 0; nByte < nNum; nByte++)
        qsData += QString("%1 ").arg(
            (uint)(unsigned char)aData[nByte], 2, 16, QLatin1Char('0'));
    qsData += '"';
    for (int nByte = 0; nByte < nNum; nByte++)
        {
        QChar oChar = QChar::fromLatin1(aData[nByte]);
        qsData += (oChar.isPrint())? oChar: QLatin1Char('.');
        }
    qsData += '"';
    return qsData;
    }


/*!
    Sleep \a nMsecs milliseconds

    \note: Use with care: will stall the GUI while sleeping when called from GUI thread
*/

void McuSerial::msleep(int nMsecs)
    {
#if 0
    // QThread::msleep is protected ...
    QThread::msleep(50);
#elif 0
    // ... so trick around it by timing out on a fake QWaitCondition.
    QMutex oLock;
    oLock.lock();
    QWaitCondition oWait;
    oWait.wait(&oLock, nMsecs);
    oLock.unlock();
#else
    // ... or do it even simpler by lifting msleep to public:
    Sleeper::msleep(nMsecs);
#endif
    }

