
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2007 by Wolfgang Hoffmann <woho@woho.de>                *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/


#ifndef ASPECT_MCUPROXY_H
#define ASPECT_MCUPROXY_H


#include "data.h"
#include "mcuserial.h"

#include <QByteArray>
#include <QObject>
#include <QString>
#include <QTime>


class McuDataFirmware
    {
public:
    void setFirmware(const QByteArray &aFirmware);
    void setUnknown() { m_aFirmware.clear(); };
    bool unknown() const { return (m_aFirmware.size() == 0); };
    QString firmware() const { return QString::fromLatin1(m_aFirmware); };
    int version() const { return m_nMajor * 100 + m_nMinor; };
private:
    QByteArray m_aFirmware;
    int m_nMajor;
    int m_nMinor;
    };

class McuDataKeys
    {
public:
    void setSerial(int nSerial) { m_nSerial = nSerial; };
    void setKeyGoto(bool bGoto) { m_bKeyGoto = bGoto; };
    void setKeyIntelly(bool bIntelly) { m_bKeyIntelly = bIntelly; };
    int serial() const { return m_nSerial; };
    bool keyGoto() const { return m_bKeyGoto; };
    bool keyIntelly() const { return m_bKeyIntelly; };
private:
    int m_nSerial;
    bool m_bKeyGoto;
    bool m_bKeyIntelly;
    };

class McuDataBacklash
    {
public:
    enum MethodRa { noneRa = 0, finishLeft = 1, finishRight = 2 };
    enum MethodDe { noneDe = 0, finishUp = 1, finishDown = 2 };
    void setMethodRa(MethodRa eMethod) { m_eMethodRa = eMethod; };
    void setValueRa(int nValue) { m_nValueRa = nValue; };
    void setMethodDe(MethodDe eMethod) { m_eMethodDe = eMethod; };
    void setValueDe(int nValue) { m_nValueDe = nValue; };
    MethodRa methodRa() const { return m_eMethodRa; };
    int valueRa() const { return m_nValueRa; };
    MethodDe methodDe() const { return m_eMethodDe; };
    int valueDe() const { return m_nValueDe; };
    QString backlash() const;
private:
    MethodRa m_eMethodRa;
    int m_nValueRa;
    MethodDe m_eMethodDe;
    int m_nValueDe;
    };

class McuDataGears
    {
public:
    McuDataGears();
    McuDataGears(
        int nRaWormWheel, float fRaGearRatio, float fRaMotorStepAngle,
        int nDeWormWheel, float fDeGearRatio, float fDeMotorStepAngle);

    void setRaWormWheel(int nWormWheel);
    void setRaGearRatio(float fGearRatio);
    void setRaMotorStepAngle(float fMotorStepAngle);
    void setDeWormWheel(int nWormWheel);
    void setDeGearRatio(float fGearRatio);
    void setDeMotorStepAngle(float fMotorStepAngle);
    int raWormWheel() const { return m_nRaWormWheel; };
    float raGearRatio() const { return m_fRaGearRatio; };
    float raMotorStepAngle() const { return m_fRaMotorStepAngle; };
    int deWormWheel() const { return m_nDeWormWheel; };
    float deGearRatio() const { return m_fDeGearRatio; };
    float deMotorStepAngle() const { return m_fDeMotorStepAngle; };

    void setClockDivSid(int nClockDivSid) { m_nClockDivSid = nClockDivSid; };
    void setClockAdjSid(int nClockAdjSid) { m_nClockAdjSid = nClockAdjSid; };
    void setClockDivSol(int nClockDivSol) { m_nClockDivSol = nClockDivSol; };
    void setClockAdjSol(int nClockAdjSol) { m_nClockAdjSol = nClockAdjSol; };
    void setClockDivLun(int nClockDivLun) { m_nClockDivLun = nClockDivLun; };
    void setClockAdjLun(int nClockAdjLun) { m_nClockAdjLun = nClockAdjLun; };
    void setCountGotoRa(int nCountGotoRa) { m_nCountGotoRa = nCountGotoRa; };
    void setCountGotoDe(int nCountGotoDe) { m_nCountGotoDe = nCountGotoDe; };
    void setCountPec(int nCountPec) { m_nCountPec = nCountPec; };
    int clockDivSid() const { return m_nClockDivSid; };
    int clockAdjSid() const { return m_nClockAdjSid; };
    int clockDivSol() const { return m_nClockDivSol; };
    int clockAdjSol() const { return m_nClockAdjSol; };
    int clockDivLun() const { return m_nClockDivLun; };
    int clockAdjLun() const { return m_nClockAdjLun; };
    int countGotoRa() const { return m_nCountGotoRa; };
    int countGotoDe() const { return m_nCountGotoDe; };
    int countPec() const { return m_nCountPec; };

    bool operator==(const McuDataGears &oRef) const;
    bool operator!=(const McuDataGears &oRef) const;

private:
    void calcInternalValues();

private:
    int m_nRaWormWheel;
    float m_fRaGearRatio;
    float m_fRaMotorStepAngle;
    int m_nDeWormWheel;
    float m_fDeGearRatio;
    float m_fDeMotorStepAngle;
    int m_nClockDivSid;
    int m_nClockAdjSid;
    int m_nClockDivSol;
    int m_nClockAdjSol;
    int m_nClockDivLun;
    int m_nClockAdjLun;
    int m_nCountGotoRa;
    int m_nCountGotoDe;
    int m_nCountPec;
    };

class McuDataPecState
    {
public:
    void setPec(int nPec, bool bOk) { m_nPec = nPec; m_bOk = bOk; };
    void setWormPos(double dfWormPos) { m_dfWormPos = dfWormPos; };
    void setTime(const QTime &oTime) { m_oTime = oTime; };
    void setState(const QString &qsState) { m_qsState = qsState; };
    int pec() const { return m_nPec; };
    bool ok() const { return m_bOk; };
    double wormPos() const { return m_dfWormPos; };
    QTime time() const { return m_oTime; };
    QString state() const { return m_qsState; };
private:
    int m_nPec;
    bool m_bOk;
    double m_dfWormPos;
    QTime m_oTime;
    QString m_qsState;
    };


class McuProxy : public QObject
    {
    Q_OBJECT

public:
    static McuProxy &proxy() { return g_oProxy; };

    // infrastructure
    bool open(const QString &qsPort, bool bRaw = false);
    void close();

    QByteArray mcuRawCommand(const QByteArray &aCommand);

    // static read-only properties; queried once and cached
    const McuDataFirmware &getFirmware() const;
    const McuDataKeys &getKeys() const;

    // cached read-write properties that need to be stored in EEPROM if changed
    enum Rotation { original = 0, reverse = 1 };
    enum Orientation { left = 0, right = 1 };
    enum Hemisphere { southern = 0, northern = 1 };
    Rotation getRotation() const;
    bool setRotation(Rotation eRotation);
    Orientation getOrientation() const;
    bool setOrientation(Orientation eOrientation);
    Hemisphere getHemisphere() const;
    McuDataGears &getGears();
    bool setGears(const McuDataGears &oGears);

    // dynamic properties; not cached, get and set directly from MCU
    enum Tracking { sidereal = 0, solar = 1, lunar = 2, off = 3 };
    Tracking getTrackingRate();
    bool setTrackingRate(Tracking eRate);

    McuDataPecState getPecState();
    bool setPec(bool bOn) const;
    bool getPecTable(PecData &rData, IProgress *pProgress = 0);
    bool putPecTable(const PecData &oData, IProgress *pProgress = 0);

    enum MoveDirection { north, south, east, west };
    enum Speed { guide_x2, move_x8, slew_x16 };
    enum RateSlow { rate1x33, rate1x66, rate2x };
    enum RateFast { rate4x, rate8x, rate16x, rate20x, rate24x,
        rate30x, rate40x, rate48x, rate60x, rate80x, rate120x };
/*
    QString guideRate() const;
    QString paddle8Rate() const;
    QString paddle16Rate() const;
    QString gotoRate() const;
    int trackingRate() const;
    int rampSteepness() const;
*/
    bool getIntellyTrack();
    bool setIntellyTrack(bool bOn);

    void guideOff() { mcuRawCommand(QString::fromUtf8("#:Q#").toLatin1()); };
    void guideEast() { mcuRawCommand(QString::fromUtf8("#:Me#").toLatin1()); };
    void guideWest() { mcuRawCommand(QString::fromUtf8("#:Mw#").toLatin1()); };

signals:
    void status(bool bOpen, McuSerial::Status eStatus) const;
    void firmware(QString qsFirmware) const;
    void busy(bool bBusy) const;

private:
    McuProxy();
    ~McuProxy();

    bool mcuAck();

    bool mcuRecvFirmware() const;
    bool mcuRecvKeys();
    bool mcuRecvSettings();
    bool mcuRecvGears();
    bool mcuRecvPecState();

    bool mcuSendRotation() const;
    bool mcuSendOrientation() const;
    bool mcuSendGears() const;
    bool mcuSendTrackingRate(Tracking eRate) const;
    bool mcuSendPec(bool bOn) const;
    bool mcuSendPecTeaching(bool bOn) const;
    bool mcuSendPecFactor(int nPecFactor, int nPecAggress) const;
    bool mcuSendIntelly(bool bOn) const;
    bool mcuSendSaveToEEPROM() const;

    static Rotation parseRotation(const QByteArray &oAnsGetSettings);

    static RateSlow decodeRateSlow(int nAnswerByte, RateSlow eDefault);
    static RateFast decodeRateFast(int nAnswerByte, RateFast eDefault);
    static Tracking decodeTrackingRate(int nAnswerByte, Tracking eDefault);

private:
    static McuProxy g_oProxy;

    McuSerial m_oPort;

    McuDataFirmware m_oFirmware;
    McuDataKeys m_oKeys;

    McuDataBacklash m_oBacklash;
    RateSlow m_eRateGuide;
    RateFast m_eRateCenter;
    RateFast m_eRateSlew;
    RateFast m_eRateGoto;
    Rotation m_eRotation;
    Orientation m_eOrientation;
    Hemisphere m_eHemisphere;
    Tracking m_eTrackingRate;
    int m_nRampSteepness;

    McuDataGears m_oGears;

    McuDataPecState m_oPecState;
    int m_nPecFactor;
    int m_nPecAggress;
    bool m_bIntellyTrack;
    };


#endif // ASPECT_MCUPROXY_H

