
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2007 by Wolfgang Hoffmann <woho@woho.de>                *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/


#include "mcucmd.h"
#include "mcuproxy.h"
#include "mcuserial.h"
#include "log.h"

#include <QtCore>




/********************************************************************/
/*!
    \class McuProxy
*/

McuProxy McuProxy::g_oProxy;


McuProxy::McuProxy()
    {
    connect(&m_oPort, SIGNAL(status(bool, McuSerial::Status)),
        this, SIGNAL(status(bool, McuSerial::Status)));
    }


McuProxy::~McuProxy()
    {
    }


bool McuProxy::open(const QString &qsPort, bool bRaw)
    {
    emit firmware(QString());
    emit busy(true);
    if (!m_oPort.open(qsPort))
        {
        emit busy(false);
        return false;
        }
    if (bRaw)
        return true;
    if (!mcuAck())
        {
        m_oPort.close();
        emit busy(false);
        return false;
        }
    return true;
    }


void McuProxy::close()
    {
    m_oFirmware.setUnknown();
    m_oPort.close();
    emit busy(false);
    }


QByteArray McuProxy::mcuRawCommand(const QByteArray &aCommand)
    {
    McuCmdStd oCmd("RawCommand", Cmd().raw(aCommand));
    if (!oCmd.send(m_oPort, 500))
        return QByteArray();
    else
        return oCmd.answer();
    }


const McuDataFirmware &McuProxy::getFirmware() const
    {
    mcuRecvFirmware();
    return m_oFirmware;
    }


const McuDataKeys &McuProxy::getKeys() const
    {
    const_cast<McuProxy *>(this)->mcuRecvKeys();
    return m_oKeys;
    }


McuProxy::Rotation McuProxy::getRotation() const
    {
    const_cast<McuProxy *>(this)->mcuRecvSettings();
    return m_eRotation;
    }


bool McuProxy::setRotation(McuProxy::Rotation eRotation)
    {
    m_eRotation = eRotation;
    return mcuSendRotation();
    }


McuProxy::Orientation McuProxy::getOrientation() const
    {
    const_cast<McuProxy *>(this)->mcuRecvSettings();
    return m_eOrientation;
    }


bool McuProxy::setOrientation(McuProxy::Orientation eOrientation)
    {
    m_eOrientation = eOrientation;
    return mcuSendOrientation();
    }


McuProxy::Hemisphere McuProxy::getHemisphere() const
    {
    const_cast<McuProxy *>(this)->mcuRecvSettings();
    return m_eHemisphere;
    }


McuDataGears &McuProxy::getGears()
    {
    const_cast<McuProxy *>(this)->mcuRecvGears();
    return m_oGears;
    }


bool McuProxy::setGears(const McuDataGears &oGears)
    {
    m_oGears = oGears;
    return mcuSendGears();
    }


McuProxy::Tracking McuProxy::getTrackingRate()
    {
    mcuRecvSettings();
    return m_eTrackingRate;
    }


bool McuProxy::setTrackingRate(McuProxy::Tracking eRate)
    {
    return mcuSendTrackingRate(eRate);
    }


McuDataPecState McuProxy::getPecState()
    {
    mcuRecvPecState();
    return m_oPecState;
    }


bool McuProxy::setPec(bool bOn) const
    {
    return mcuSendPec(bOn);
    }


bool McuProxy::getPecTable(PecData &rData, IProgress *pProgress)
    {
    mcuRecvPecState();
    int nFactor = m_nPecFactor;
    int nAggress = m_nPecAggress;

    QVector<PecSample::Val> aTable;
    aTable.resize(PecData::c_nRowNum);
    m_oPort.flush();
    int nRowSend = 0;
    int nRowRecv = 0;
    while (nRowRecv < PecData::c_nRowNum)
        {
        if (nRowSend < PecData::c_nRowNum)
            {
            McuCmdVoid oCmd("GetPecValue", Cmd("#").num(nRowSend).str(":VR#"));
            if (!m_oPort.send(&oCmd))
                break;
            nRowSend += 1;
            }

        // while at sending, don't wait for answer bytes
        // if sending is complete, wait for remaining answers
        // wait one by one to keep progress indicator running
        int nRemain = PecData::c_nRowNum - nRowRecv;
        McuAns oAns("GetPecValue", (nRowSend < PecData::c_nRowNum)? 0: 1, nRemain);
        if (!m_oPort.recv(&oAns))
            break;
        for (int nByte = 0; nByte < oAns.answer().size(); nByte++)
            aTable[nRowRecv++] = PecSample::int2Val(oAns.byte(nByte) - '0');

        if (pProgress != 0)
            if (pProgress->setValue((double)nRowRecv / (double)PecData::c_nRowNum))
                break;
        }
    int nRemain = nRowSend - nRowRecv;
    if (nRemain > 0)
        {
        // if interrupted, flush pending answers.
        // don't use flush() here, as it won't wait
        McuAns oAns("GetPecValue", nRemain, nRemain);
        m_oPort.recv(&oAns);
        }
    if (nRowRecv < PecData::c_nRowNum)
        return false;
    if (pProgress != 0)
        pProgress->setValue(1.);

    return rData.set(nFactor, nAggress, aTable);
    }


bool McuProxy::putPecTable(const PecData &oData, IProgress *pProgress)
    {
    double dfProgress = 0.;
    if (pProgress != 0)
        pProgress->setValue(dfProgress);
    mcuRecvPecState();
    dfProgress += .01;
    if (pProgress != 0)
        pProgress->setValue(dfProgress);
    if (!m_oPecState.ok())
        {
        // if PEC state is not ok, clear it by starting a minimal PEC training
        mcuSendPecTeaching(true);
        mcuRecvPecState();
        McuSerial::msleep(200);
        dfProgress += .01;
        if (pProgress != 0)
            pProgress->setValue(dfProgress);
        mcuSendPecTeaching(false);
        McuSerial::msleep(200);
        dfProgress += .01;
        if (pProgress != 0)
            pProgress->setValue(dfProgress);
        // after stopping PEC training, must wait quite some time before
        // sending commands like querying PEC state, otherwise MCU gets pretty confused
        // so send a ACK to see when MCU is ready
        McuCmdFixed oCmd("Ack", Cmd().byte(0x06), 1);
        if (!oCmd.send(m_oPort, 2000))
            return false;
        mcuRecvPecState();
        }
    dfProgress += .01;
    if (pProgress != 0)
        pProgress->setValue(dfProgress);
    if (!mcuSendPecFactor(oData.factor(), oData.aggress()))
        return false;
    dfProgress += .01;
    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        {
        if (pProgress != 0)
            if (pProgress->setValue(dfProgress +
                (0.99 - dfProgress) * (double)nRow / (double)PecData::c_nRowNum))
                return false;
        int nVal = oData.valueInt(nRow);
        McuCmdVoid oCmd("SetPecValue", Cmd("#").num(nRow).str(":XR").num(nVal).str("#"));
        // be fast: just send, don't flush before or receive afterwards
        if (!m_oPort.send(&oCmd))
            return false;
        }
    if (pProgress != 0)
        pProgress->setValue(.99);
    McuCmdVoid oCmd("SavePecTable", "#:XYR#");
    if (!oCmd.send(m_oPort))
        return false;
    if (pProgress != 0)
        pProgress->setValue(1.);

    // todo: verify table by reading it back in?
    return true;
    }


bool McuProxy::getIntellyTrack()
    {
    mcuRecvPecState();
    return m_bIntellyTrack;
    }


bool McuProxy::setIntellyTrack(bool bOn)
    {
    return mcuSendIntelly(bOn);
    }


bool McuProxy::mcuAck()
    {
    McuCmdFixed oCmd("ACK", Cmd().byte('\x06'), 1);
    if ((!oCmd.send(m_oPort)) ||
        (oCmd.answer().size() == 0) ||
        (oCmd.byte(0) != 'P'))
        return false;
    return true;
    }


bool McuProxy::mcuRecvFirmware() const
    {
    McuProxy *that = const_cast<McuProxy *>(this);
    that->m_oFirmware.setUnknown();
    McuCmdQueryUnlimited oCmd("GetFirmware", "#:GV#", 1);
    if ((!oCmd.send(m_oPort)) || (oCmd.answer().size() == 0))
        return false;
    that->m_oFirmware.setFirmware(oCmd.answer());
    emit firmware(m_oFirmware.firmware());
    return true;
    }


bool McuProxy::mcuRecvKeys()
    {
    if ((m_oFirmware.unknown()) && (!mcuRecvFirmware()))
        return false;
    m_oKeys.setSerial(0);
    m_oKeys.setKeyGoto(false);
    m_oKeys.setKeyIntelly(false);
    McuCmdFixed oCmd("GetKeys", Cmd().str("#").byte('\xfe'), 4);
    if (!oCmd.send(m_oPort))
        return false;
    // serial number has reversed byte order (big endian)
    int nSerial = (oCmd.byte(2) << 8) + oCmd.byte(3);
    m_oKeys.setSerial(nSerial & 0x3fff);
    if (m_oFirmware.version() <= 359)
        {
        }
    else
        {
        m_oKeys.setKeyGoto((nSerial & 0x8000) != 0);
        m_oKeys.setKeyIntelly((nSerial & 0x4000) != 0);
        }
    return true;
    }


bool McuProxy::mcuRecvSettings()
    {
    if ((m_oFirmware.unknown()) && (!mcuRecvFirmware()))
        return false;
    if (m_oFirmware.version() <= 359)
        {
        McuCmdFixed oCmdSetL("GetSettingsLow", "<#:VX#", 12);
        if (!oCmdSetL.send(m_oPort))
            {
            // todo: do proper error-handling (set members to defined values!)
            return false;
            }

        // with firmware v3.59, only do the bare minimum required for Aspect
        m_eRotation = ((oCmdSetL.byte(10) & 0x0f) == 0x00)? original: reverse;
        m_eOrientation = ((oCmdSetL.byte(10) & 0xf0) == 0x00)? left: right;
        m_eTrackingRate = decodeTrackingRate(oCmdSetL.byte(11), off);
        return true;
        }
    else
        {
        McuCmdFixed oCmdSet("GetSettings", "#0:VX#", 13);
        if (!oCmdSet.send(m_oPort))
            {
            // todo: do proper error-handling (set members to defined values!)
            return false;
            }

        m_oBacklash.setMethodRa(
            (oCmdSet.byte(0) == 0xf0)? McuDataBacklash::finishLeft:
            (oCmdSet.byte(0) == 0x00)? McuDataBacklash::noneRa:
            (oCmdSet.byte(0) == 0x0f)? McuDataBacklash::finishRight: McuDataBacklash::noneRa);
        m_oBacklash.setValueRa(oCmdSet.word(1));
        m_oBacklash.setMethodDe(
            (oCmdSet.byte(3) == 0xf0)? McuDataBacklash::finishDown:
            (oCmdSet.byte(3) == 0x00)? McuDataBacklash::noneDe:
            (oCmdSet.byte(3) == 0x0f)? McuDataBacklash::finishUp: McuDataBacklash::noneDe);
        m_oBacklash.setValueDe(oCmdSet.word(4));
        m_eRateGuide = decodeRateSlow(oCmdSet.byte(6), rate2x); // aka. paddle 2x
        m_eRateCenter = decodeRateFast(oCmdSet.byte(7), rate8x); // aka. paddle 8x
        m_eRateSlew = decodeRateFast(oCmdSet.byte(8), rate16x); // aka. paddle 16x
        m_eRateGoto = decodeRateFast(oCmdSet.byte(9), rate20x);
        m_eRotation = ((oCmdSet.byte(10) & 0x80) == 0x00)? original: reverse;
        m_eOrientation = ((oCmdSet.byte(10) & 0x02) == 0x00)? left: right;
        m_eHemisphere = ((oCmdSet.byte(10) & 0x10) == 0x00)? southern: northern;
        m_eTrackingRate = decodeTrackingRate(oCmdSet.byte(11), off);
        m_nRampSteepness = oCmdSet.byte(12);
        return true;
        }
    }


bool McuProxy::mcuRecvGears()
    {
    if ((m_oFirmware.unknown()) && (!mcuRecvFirmware()))
        return false;
    if (m_oFirmware.version() <= 359)
        {
        McuCmdFixed oCmdSetH("GetSettingsHigh", ">#:VX#", 6);
        if (!oCmdSetH.send(m_oPort))
            {
            // todo: do proper error-handling (set members to defined values!)
            return false;
            }

        // with firmware v3.59, only do the bare minimum required for Aspect
        m_oGears.setRaWormWheel(oCmdSetH.word(4));
        return true;
        }
    else
        {
        McuCmdFixed oCmdSid("GetClockSidereal", "#1:VX#", 5);
        McuCmdFixed oCmdSol("GetClockSolar", "#2:VX#", 5);
        McuCmdFixed oCmdLun("GetClockLunar", "#3:VX#", 5);
        McuCmdFixed oCmdGoto("GetCountGoto", "#4:VX#", 4);
        McuCmdFixed oCmdPec("GetCountPec", "#5:VX#", 2);
        McuCmdFixed oCmdRa("GetMountRa", "#6:VX#", 10);
        McuCmdFixed oCmdDe("GetMountDe", "#7:VX#", 10);
        if (!(
            oCmdSid.send(m_oPort) &&
            oCmdSol.send(m_oPort) &&
            oCmdLun.send(m_oPort) &&
            oCmdGoto.send(m_oPort) &&
            oCmdPec.send(m_oPort) &&
            oCmdRa.send(m_oPort) &&
            oCmdDe.send(m_oPort)))
            {
            // todo: do proper error-handling (set members to defined values!)
            return false;
            }

        m_oGears.setRaWormWheel(oCmdRa.word(0));
        m_oGears.setRaGearRatio(oCmdRa.real(2));
        m_oGears.setRaMotorStepAngle(oCmdRa.real(6));
        m_oGears.setDeWormWheel(oCmdDe.word(0));
        m_oGears.setDeGearRatio(oCmdDe.real(2));
        m_oGears.setDeMotorStepAngle(oCmdDe.real(6));
        m_oGears.setClockDivSid(oCmdSid.word(0));
        m_oGears.setClockAdjSid(((oCmdSid.byte(2) == 1)? -1: 1) * oCmdSid.word(3));
        m_oGears.setClockDivSol(oCmdSol.word(0));
        m_oGears.setClockAdjSol(((oCmdSol.byte(2) == 1)? -1: 1) * oCmdSol.word(3));
        m_oGears.setClockDivLun(oCmdLun.word(0));
        m_oGears.setClockAdjLun(((oCmdLun.byte(2) == 1)? -1: 1) * oCmdLun.word(3));
        m_oGears.setCountGotoRa(oCmdGoto.word(0));
        m_oGears.setCountGotoDe(oCmdGoto.word(2));
        m_oGears.setCountPec(oCmdPec.word(0));
        return true;
        }
    }


bool McuProxy::mcuRecvPecState()
    {
    McuCmdFixed oCmd("GetPecState", "#:VS#", 10);
    m_oPecState.setTime(QTime::currentTime());
    if (!oCmd.send(m_oPort))
        {
        m_oPecState.setPec(5, false);
        m_oPecState.setWormPos(0.);
        m_oPecState.setState(QString::fromUtf8("unknown"));
        m_nPecFactor = 0;
        m_nPecAggress = 0;
        m_bIntellyTrack = false;
        return false;
        }

    bool bOk = ((oCmd.byte(5) & 0x80) != 0);
    bool bOnHandbox = ((oCmd.byte(5) & 0x01) != 0);
    bool bOnSerial = ((oCmd.byte(5) & 0x02) != 0);
    bool bTraining = ((oCmd.byte(5) & 0x04) != 0);
    m_oPecState.setPec((!bOk)? 4: bTraining? 3: bOnHandbox? 2: bOnSerial? 1: 0, bOk);
    int nPecDiv = oCmd.word(1);
    int nPecPos = oCmd.word(3);
    int nPecByte = oCmd.word(6);
    int nPecBit = oCmd.word(7);
    int nPecPoint = ((nPecByte & 0x3f) << 2) + (nPecBit & 0x3);
    m_oPecState.setWormPos((double)nPecPoint + (double)nPecPos / (double)nPecDiv);
    m_oPecState.setState(QString::fromUtf8("%1, %2%3%4 %5 %6 %7 %8 %9 %10 %11 %12")
        .arg(QString::fromUtf8(bOk? "ok": "not ok"))
        .arg(QString::fromUtf8(bTraining? "teach": (bOnHandbox || bOnSerial)? "on": "off"))
        .arg(QString::fromUtf8(bOnHandbox? " by hand": ""))
        .arg(QString::fromUtf8(bOnSerial? " by serial": ""))
        .arg(oCmd.byte(0), 2, 16, QLatin1Char('0'))
        .arg(oCmd.word(1), 4, 16, QLatin1Char('0'))
        .arg(oCmd.word(3), 4, 16, QLatin1Char('0'))
        .arg(oCmd.byte(5), 2, 16, QLatin1Char('0'))
        .arg(oCmd.byte(6), 2, 16, QLatin1Char('0'))
        .arg(oCmd.byte(7), 2, 16, QLatin1Char('0'))
        .arg(oCmd.byte(8), 2, 16, QLatin1Char('0'))
        .arg(oCmd.byte(9), 2, 16, QLatin1Char('0')));
    m_nPecFactor = (oCmd.byte(9) & 0x0f);
    m_nPecAggress = (oCmd.byte(9) & 0xf0) >> 4;
    m_bIntellyTrack = ((oCmd.byte(0) & 0x03) != 0);

    return true;
    }


bool McuProxy::mcuSendRotation() const
    {
    const char *szCmd = 0;
    switch (m_eRotation)
        {
        case original: szCmd = "#<x"; break;
        case reverse:  szCmd = "#>x"; break;
        };
    McuCmdStd oCmd("SetRotation", szCmd);
    return oCmd.send(m_oPort);
    }


bool McuProxy::mcuSendOrientation() const
    {
    const char *szCmd = 0;
    switch (m_eOrientation)
        {
        case left:  szCmd = "#>X"; break;
        case right: szCmd = "#<X"; break;
        };
    McuCmdStd oCmd("SetOrientation", szCmd);
    return oCmd.send(m_oPort);
    }


bool McuProxy::mcuSendGears() const
    {
    // set values
    McuCmdStd oCmdRaWorm("SetMountRaWorm", Cmd("#:ZWR").word(m_oGears.raWormWheel()).str("#"));
    McuCmdStd oCmdRaGear("SetMountRaGear", Cmd("#:ZGR").real(m_oGears.raGearRatio()).str("#"));
    McuCmdStd oCmdRaStep("SetMountRaStep", Cmd("#:ZMR").real(m_oGears.raMotorStepAngle()).str("#"));
    McuCmdStd oCmdDecWorm("SetMountDecWorm", Cmd("#:ZWD").word(m_oGears.deWormWheel()).str("#"));
    McuCmdStd oCmdDecGear("SetMountDecGear", Cmd("#:ZGD").real(m_oGears.deGearRatio()).str("#"));
    McuCmdStd oCmdDecStep("SetMountDecStep", Cmd("#:ZMD").real(m_oGears.deMotorStepAngle()).str("#"));
    McuCmdStd oCmdClockDivSid("SetClockDivSidereal", Cmd("#").num(m_oGears.clockDivSid()).str(":ZDS#"));
    McuCmdStd oCmdClockAdjSid("SetClockAdjSidereal", Cmd("#").num(m_oGears.clockAdjSid()).str(":ZAS#"));
    McuCmdStd oCmdClockDivSol("SetClockDivSolar", Cmd("#").num(m_oGears.clockDivSol()).str(":ZDs#"));
    McuCmdStd oCmdClockAdjSol("SetClockAdjSolar", Cmd("#").num(m_oGears.clockAdjSol()).str(":ZAs#"));
    McuCmdStd oCmdClockDivLun("SetClockDivLunar", Cmd("#").num(m_oGears.clockDivLun()).str(":ZDL#"));
    McuCmdStd oCmdClockAdjLun("SetClockAdjLunar", Cmd("#").num(m_oGears.clockAdjLun()).str(":ZAL#"));
    McuCmdStd oCmdCountGotoRa("SetCountGotoRa", Cmd("#").num(m_oGears.countGotoRa()).str(":ZgR#"));
    McuCmdStd oCmdCountGotoDec("SetCountGotoDec", Cmd("#").num(m_oGears.countGotoDe()).str(":ZgD#"));
    McuCmdStd oCmdCountPec("SetCountPec", Cmd("#").num(m_oGears.countPec()).str(":ZPE#"));
    if (!(
        oCmdRaWorm.send(m_oPort) &&
        oCmdRaGear.send(m_oPort) &&
        oCmdRaStep.send(m_oPort) &&
        oCmdDecWorm.send(m_oPort) &&
        oCmdDecGear.send(m_oPort) &&
        oCmdDecStep.send(m_oPort) &&
        oCmdClockDivSid.send(m_oPort) &&
        oCmdClockAdjSid.send(m_oPort) &&
        oCmdClockDivSol.send(m_oPort) &&
        oCmdClockAdjSol.send(m_oPort) &&
        oCmdClockDivLun.send(m_oPort) &&
        oCmdClockAdjLun.send(m_oPort) &&
        oCmdCountGotoRa.send(m_oPort) &&
        oCmdCountGotoDec.send(m_oPort) &&
        oCmdCountPec.send(m_oPort)))
        return false;

    // read back values to verify if written correctly
    McuCmdFixed oCmdSid("GetClockSidereal", "#1:VX#", 5);
    McuCmdFixed oCmdSol("GetClockSolar", "#2:VX#", 5);
    McuCmdFixed oCmdLun("GetClockLunar", "#3:VX#", 5);
    McuCmdFixed oCmdGoto("GetCountGoto", "#4:VX#", 4);
    McuCmdFixed oCmdPec("GetCountPec", "#5:VX#", 2);
    if (!(
        oCmdSid.send(m_oPort) &&
        oCmdSol.send(m_oPort) &&
        oCmdLun.send(m_oPort) &&
        oCmdGoto.send(m_oPort) &&
        oCmdPec.send(m_oPort)))
        {
        // todo: do proper error-handling
        return false;
        }
    if (!(
        (m_oGears.clockDivSid() == oCmdSid.word(0)) &&
        (m_oGears.clockAdjSid() == ((oCmdSid.byte(2) == 1)? -1: 1) * oCmdSid.word(3)) &&
        (m_oGears.clockDivSol() == oCmdSol.word(0)) &&
        (m_oGears.clockAdjSol() == ((oCmdSol.byte(2) == 1)? -1: 1) * oCmdSol.word(3)) &&
        (m_oGears.clockDivLun() == oCmdLun.word(0)) &&
        (m_oGears.clockAdjLun() == ((oCmdLun.byte(2) == 1)? -1: 1) * oCmdLun.word(3)) &&
        (m_oGears.countGotoRa() == oCmdGoto.word(0)) &&
        (m_oGears.countGotoDe() == oCmdGoto.word(2)) &&
        (m_oGears.countPec() == oCmdPec.word(0))))
        {
        LOG(0, "McuProxy::mcuSendGears(): mismatch while verifying:\n"
            "    SidDiv,SidAdj %d,%d: %d,%d\n"
            "    SolDiv,SolAdj %d,%d: %d,%d\n"
            "    LunDiv,LunAdj %d,%d: %d,%d\n"
            "    CountGoto %d,%d: %d,%d\n"
            "    CountPec %d: %d",
            m_oGears.clockDivSid(), m_oGears.clockAdjSid(), oCmdSid.word(0),
            ((oCmdSid.byte(2) == 1)? -1: 1) * oCmdSid.word(3),
            m_oGears.clockDivSol(), m_oGears.clockAdjSol(), oCmdSol.word(0),
            ((oCmdSol.byte(2) == 1)? -1: 1) * oCmdSol.word(3),
            m_oGears.clockDivLun(), m_oGears.clockAdjLun(), oCmdLun.word(0),
            ((oCmdLun.byte(2) == 1)? -1: 1) * oCmdLun.word(3),
            m_oGears.countGotoRa(), m_oGears.countGotoDe(), oCmdGoto.word(0), oCmdGoto.word(2),
            m_oGears.countPec(), oCmdPec.word(0));
        return false;
        }

    if (!mcuSendSaveToEEPROM())
        return false;

    return true;
    }


bool McuProxy::mcuSendTrackingRate(McuProxy::Tracking eRate) const
    {
    const char *szCmd = 0;
    switch (eRate)
        {
        case off:      szCmd = "#-1T"; break;
        case sidereal: szCmd = "#0T"; break;
        case solar:    szCmd = "#1T"; break;
        case lunar:    szCmd = "#2T"; break;
        };
    McuCmdStd oCmd("SetTrackingRate", szCmd);
    if (!oCmd.send(m_oPort))
        return false;
    return true;
    }


bool McuProxy::mcuSendPec(bool bOn) const
    {
    const char *szCmd = 0;
    switch (bOn)
        {
        case false: szCmd = "#0P"; break;
        case true:  szCmd = "#1P"; break;
        };
    McuCmdStd oCmd("SetPec", szCmd);
    if (!oCmd.send(m_oPort))
        return false;
    return true;
    }


bool McuProxy::mcuSendPecTeaching(bool bOn) const
    {
    const char *szCmd = 0;
    switch (bOn)
        {
        case false: szCmd = "#-0P"; break;
        case true:  szCmd = "#-1P"; break;
        };
    McuCmdStd oCmd("SetPecTeaching", szCmd);
    if (!oCmd.send(m_oPort))
        return false;
    return true;
    }


bool McuProxy::mcuSendPecFactor(int nPecFactor, int nPecAggress) const
    {
    if ((m_oFirmware.unknown()) && (!mcuRecvFirmware()))
        return false;
    int nValue = 1;
    if (m_oFirmware.version() <= 359)
        {
        if (nPecFactor < 1)
            nPecFactor = 1;
        else if (nPecFactor > 9)
            nPecFactor = 9;
        nValue = nPecFactor;
        }
    else
        {
        if (nPecFactor < 1)
            nPecFactor = 1;
        else if (nPecFactor > 15)
            nPecFactor = 15;
        if (nPecAggress < 1)
            nPecAggress = 1;
        else if (nPecAggress > 15)
            nPecAggress = 15;
        nValue = nPecFactor | (nPecAggress << 4);
        }
    McuCmdStd oCmdPec("SetPecFactor", Cmd("#").num(nValue).str("p#"));
    if (!oCmdPec.send(m_oPort))
        return false;
    /* don't save to EEPROM for now, as it seems to damage PEC count
    // it seems that PEC factor is save to EEPROM while (auto-)parking
    // still, save to EEPROM to be protocol conform
    if (!mcuSendSaveToEEPROM())
        return false;
    */
    return true;
    }


bool McuProxy::mcuSendIntelly(bool bOn) const
    {
    const char *szCmd = 0;
    switch (bOn)
        {
        case false: szCmd = "#<I"; break;
        case true:  szCmd = "#>I"; break;
        };
    // if successful, SetIntellyTrack answers '?' when switched off, and '!' when on
    McuCmdBool oCmd("SetIntellyTrack", szCmd, bOn? '!': '?');
    if (!oCmd.send(m_oPort))
        return false;
    if (!oCmd.success())
        LOG(0, "McuDataPecState::mcuSet(): SetIntellyTrack %s returned error \"%s\"",
            bOn? "on": "off", LQS(oCmd.errorReason()));
    // hack:
    // get sidereal time to avoid troubles with certain following commands
    // (such as <#:VX#)
    McuCmdString oDummy("GetSiderealTime", "#:GS#");
    oDummy.send(m_oPort);
    return true;
    }


bool McuProxy::mcuSendSaveToEEPROM() const
    {
    if ((m_oFirmware.unknown()) && (!mcuRecvFirmware()))
        return false;
    if (m_oFirmware.version() <= 359)
        {
        McuCmdString oCmdSave("SaveToEEPROM", "#>-9595Y");
        if ((!oCmdSave.send(m_oPort, 2000)) ||
            (QString::fromLatin1(oCmdSave.answer()) != QString::fromUtf8("OK#")))
            {
            LOG(0, "McuProxy::mcuSendSaveToEEPROM(): failed to save values to EEPROM");
            return false;
            }
        }
    else
        {
        McuCmdFixed oCmdSave("SaveToEEPROM", "#>-9595Y", 1);
        if ((!oCmdSave.send(m_oPort, 2000)) || (oCmdSave.byte(0) != 0x06))
            {
            LOG(0, "McuProxy::mcuSendSaveToEEPROM(): failed to save values to EEPROM");
            return false;
            }
        if ((!oCmdSave.send(m_oPort, 2000)) || (oCmdSave.byte(0) != 0x06))
            {
            LOG(0, "McuProxy::mcuSendSaveToEEPROM(): failed to save values to EEPROM");
            return false;
            }
        if ((!oCmdSave.send(m_oPort, 2000)) || (oCmdSave.byte(0) != 0x06))
            {
            LOG(0, "McuProxy::mcuSendSaveToEEPROM(): failed to save values to EEPROM");
            return false;
            }
        }
    return true;
    }


McuProxy::RateSlow McuProxy::decodeRateSlow(int nAnswerByte, McuProxy::RateSlow eDefault)
    {
    switch (nAnswerByte)
        {
        case 0x01: return rate1x33;
        case 0x02: return rate1x66;
        case 0x03: return rate2x;
        default: return eDefault;
        }
    }

McuProxy::RateFast McuProxy::decodeRateFast(int nAnswerByte, McuProxy::RateFast eDefault)
    {
    switch (nAnswerByte)
        {
        case 0x07: return rate4x;
        case 0x08: return rate8x;
        case 0x09: return rate16x;
        case 0x0a: return rate20x;
        case 0x0b: return rate24x;
        case 0x0c: return rate30x;
        case 0x0d: return rate40x;
        case 0x0e: return rate48x;
        case 0x0f: return rate60x;
        case 0x10: return rate80x;
        case 0x11: return rate120x;
        default: return eDefault;
        }
    }

McuProxy::Tracking McuProxy::decodeTrackingRate(int nAnswerByte, McuProxy::Tracking eDefault)
    {
    switch (nAnswerByte)
        {
        case  0: return sidereal;
        case  1: return solar;
        case  2: return lunar;
        case -1: return off;
        default: return eDefault;
        }
    }
#if 0
#######################################################################

V4.00 command #x:Ssy#

Code:
  #x:Ssy#   Set speed y adjustment to the value of x

Default:
    y=0,x=1
    y=1,x=2
    y=2,x=3

Values: 
    y = 0 = 8x speed
    y = 1 = 16x speed
    y = 2 = GOTO speed

    x = 0 = 4x
    x = 1 = 8x
    x = 2 = 16x
    x = 3 = 20x
    x = 4 = 24x
    x = 5 = 30x
    x = 6 = 40x
    x = 7 = 48x
    x = 8 = 60x
    x = 9 = 80x
    x = 10 = 120x

#endif


/********************************************************************/
/*!
    \class McuDataFirmware
*/

void McuDataFirmware::setFirmware(const QByteArray &aFirmware)
    {
    m_aFirmware = aFirmware;
    m_nMajor = (m_aFirmware.size() < 1)? 0: m_aFirmware[0] - '0';
    m_nMinor = (m_aFirmware.size() < 4)? 0:
        ((m_aFirmware[2] - '0') * 10 + (m_aFirmware[3] - '0'));
    }


/********************************************************************/
/*!
    \class McuDataKeys
*/


/********************************************************************/
/*!
    \class McuDataBacklash
*/

QString McuDataBacklash::backlash() const
    {
    return QString::fromUtf8("RA %1, %2; Dec %3, %4")
        .arg(QString::fromUtf8(
            (m_eMethodRa == noneRa)?  "none":
            (m_eMethodRa == finishLeft)? "finish Left":
            (m_eMethodRa == finishRight)?  "finish Right": "?"))
        .arg(m_nValueRa)
        .arg(QString::fromUtf8(
            (m_eMethodDe == noneDe)?  "none":
            (m_eMethodDe == finishUp)? "finish Up":
            (m_eMethodDe == finishDown)?  "finish Down": "?"))
        .arg(m_nValueDe);
    }


/********************************************************************/
/*!
    \class McuDataGears
*/

McuDataGears::McuDataGears()
:   m_nRaWormWheel(1),
    m_fRaGearRatio(0.),
    m_fRaMotorStepAngle(0.),
    m_nDeWormWheel(1),
    m_fDeGearRatio(0.),
    m_fDeMotorStepAngle(0.),
    m_nClockDivSid(0),
    m_nClockAdjSid(0),
    m_nClockDivSol(0),
    m_nClockAdjSol(0),
    m_nClockDivLun(0),
    m_nClockAdjLun(0),
    m_nCountGotoRa(0),
    m_nCountGotoDe(0),
    m_nCountPec(0)
    {
    }

McuDataGears::McuDataGears(
    int nRaWormWheel, float fRaGearRatio, float fRaMotorStepAngle,
    int nDeWormWheel, float fDeGearRatio, float fDeMotorStepAngle)
:   m_nRaWormWheel(nRaWormWheel),
    m_fRaGearRatio(fRaGearRatio),
    m_fRaMotorStepAngle(fRaMotorStepAngle),
    m_nDeWormWheel(nDeWormWheel),
    m_fDeGearRatio(fDeGearRatio),
    m_fDeMotorStepAngle(fDeMotorStepAngle),
    m_nClockDivSid(0),
    m_nClockAdjSid(0),
    m_nClockDivSol(0),
    m_nClockAdjSol(0),
    m_nClockDivLun(0),
    m_nClockAdjLun(0),
    m_nCountGotoRa(0),
    m_nCountGotoDe(0),
    m_nCountPec(0)
    {
    calcInternalValues();
    }

void McuDataGears::setRaWormWheel(int nWormWheel)
    {
    m_nRaWormWheel = nWormWheel;
    calcInternalValues();
    }

void McuDataGears::setRaGearRatio(float fGearRatio)
    {
    m_fRaGearRatio = fGearRatio;
    calcInternalValues();
    }

void McuDataGears::setRaMotorStepAngle(float fMotorStepAngle)
    {
    m_fRaMotorStepAngle = fMotorStepAngle;
    calcInternalValues();
    }

void McuDataGears::setDeWormWheel(int nWormWheel)
    {
    m_nDeWormWheel = nWormWheel;
    calcInternalValues();
    }

void McuDataGears::setDeGearRatio(float fGearRatio)
    {
    m_fDeGearRatio = fGearRatio;
    calcInternalValues();
    }

void McuDataGears::setDeMotorStepAngle(float fMotorStepAngle)
    {
    m_fDeMotorStepAngle = fMotorStepAngle;
    calcInternalValues();
    }

bool McuDataGears::operator==(const McuDataGears &oRef) const
    {
    // todo: cope with slight variation of ClockAdj values due to rounding errors?
    return (
        (m_nRaWormWheel == oRef.m_nRaWormWheel) &&
        (m_fRaGearRatio == oRef.m_fRaGearRatio) &&
        (m_fRaMotorStepAngle == oRef.m_fRaMotorStepAngle) &&
        (m_nDeWormWheel == oRef.m_nDeWormWheel) &&
        (m_fDeGearRatio == oRef.m_fDeGearRatio) &&
        (m_fDeMotorStepAngle == oRef.m_fDeMotorStepAngle) &&
        (m_nClockDivSid == oRef.m_nClockDivSid) &&
        (m_nClockDivSid == oRef.m_nClockDivSid) &&
        (m_nClockAdjSid == oRef.m_nClockAdjSid) &&
        (m_nClockDivSol == oRef.m_nClockDivSol) &&
        (m_nClockAdjSol == oRef.m_nClockAdjSol) &&
        (m_nClockDivLun == oRef.m_nClockDivLun) &&
        (m_nClockAdjLun == oRef.m_nClockAdjLun) &&
        (m_nCountGotoRa == oRef.m_nCountGotoRa) &&
        (m_nCountGotoDe == oRef.m_nCountGotoDe) &&
        (m_nCountPec == oRef.m_nCountPec));
    }

bool McuDataGears::operator!=(const McuDataGears &oRef) const
    {
    return !operator==(oRef);
    }

void McuDataGears::calcInternalValues()
    {
    // clocks
    static const double c_dfClockGear = 92160.;
    static const int c_nClockAdjMinMax = 65534;
    static const double c_dfClockDivSid = 1378624000000.; // 16MHz MCU Clk * 86164
    static const double c_dfClockDivSol = 1382400000000.;
    static const double c_dfClockDivLun = 1428944000000.;

    double dfRaStepRatio = (double)m_nRaWormWheel * m_fRaGearRatio * (7.5 / m_fRaMotorStepAngle);
    double dfDeStepRatio = (double)m_nRaWormWheel * m_fRaGearRatio * (7.5 / m_fDeMotorStepAngle);

    double dfClockDivSid = c_dfClockDivSid / (dfRaStepRatio * c_dfClockGear);
    m_nClockDivSid = (int)(dfClockDivSid + .5);
    double dfClockRemSid = (double)m_nClockDivSid - dfClockDivSid;
    m_nClockAdjSid = 0;
    if (dfClockRemSid != 0.)
        m_nClockAdjSid = (int)(dfClockDivSid / dfClockRemSid + .5);
    if (abs(m_nClockAdjSid) >= c_nClockAdjMinMax)
        m_nClockAdjSid = 0;

    double dfClockDivSol = c_dfClockDivSol / (dfRaStepRatio * c_dfClockGear);
    m_nClockDivSol = (int)(dfClockDivSol + .5);
    double dfClockRemSol = (double)m_nClockDivSol - dfClockDivSol;
    m_nClockAdjSol = 0;
    if (dfClockRemSol != 0.)
        m_nClockAdjSol = (int)(dfClockDivSol / dfClockRemSol + .5);
    if (abs(m_nClockAdjSol) >= c_nClockAdjMinMax)
        m_nClockAdjSol = 0;

    double dfClockDivLun = c_dfClockDivLun / (dfRaStepRatio * c_dfClockGear);
    m_nClockDivLun = (int)(dfClockDivLun + .5);
    double dfClockRemLun = (double)m_nClockDivLun - dfClockDivLun;
    m_nClockAdjLun = 0;
    if (dfClockRemLun != 0.)
        m_nClockAdjLun = (int)(dfClockDivLun / dfClockRemLun + .5);
    if (abs(m_nClockAdjLun) >= c_nClockAdjMinMax)
        m_nClockAdjLun = 0;

    // goto and pec counters
    static const double c_dfCountRa = 16875000. / 7.5; // Ra is hh:mm.m
    static const double c_dfCountDe = 25312500. / 7.5; // De is dd°mm'
    m_nCountGotoRa = (int)(c_dfCountRa / dfRaStepRatio + .5);
    m_nCountGotoDe = (int)(c_dfCountDe / dfDeStepRatio + .5);
    m_nCountPec = (int)((m_fRaGearRatio * 2880. / m_fRaMotorStepAngle) / 256.);
    }


/********************************************************************/
/*!
    \class McuDataPecState
*/




/********************************************************************/
/*
*/

#if 0

/********************************************************************/
/*!
    \class McuDataIntellyTrack
*/


void McuCom::cmdSetGuideRate(RateSlow eRate)
    {
    const char *szCmd = 0;
    switch (eRate)
        {
        case x1_33: szCmd = ">#0G"; break;
        case x1_66: szCmd = "<#-1G"; break;
        case x2:    szCmd = "<#1G"; break;
        };
    sendCmd(szCmd);
    }


void McuCom::cmdSetMoveRate(RateFast eRate)
    {
    const char *szCmd = 0;
    switch (eRate)
        {
        case x4:  szCmd = "#0:Ss1#"; break;
        case x8:  szCmd = "#1:Ss1#"; break;
        case x16: szCmd = "#2:Ss1#"; break;
        case x20: szCmd = "#3:Ss1#"; break;
        case x27: szCmd = "#4:Ss1#"; break;
        case x40: szCmd = "#5:Ss1#"; break;
        };
    sendCmd(szCmd);
    }


void McuCom::cmdSetSlewRate(RateFast eRate)
    {
    const char *szCmd = 0;
    switch (eRate)
        {
        case x4:  szCmd = "#0:Ss2#"; break;
        case x8:  szCmd = "#1:Ss2#"; break;
        case x16: szCmd = "#2:Ss2#"; break;
        case x20: szCmd = "#3:Ss2#"; break;
        case x27: szCmd = "#4:Ss2#"; break;
        case x40: szCmd = "#5:Ss2#"; break;
        };
    sendCmd(szCmd);
    }


void McuCom::cmdSetSpeed(Speed eSpeed)
    {
    // choose one of the basic speed classes
    // guide (2x): 1.33x, 1.66x, 2x
    // move  (8x): 
    // slew (16x)
    const char *szCmd = 0;
    switch (eSpeed)
        {
        case guide_x2: szCmd = "#:RG#"; break;
        case move_x8:  szCmd = "#:RM#"; break;
        case slew_x16: szCmd = "#:RS#"; break;
        };
    sendCmd(szCmd);
    }


void McuCom::cmdPulseGuide(Direction eDirection, unsigned int nTimeMsecs)
    {
    if (nTimeMsecs > 9999)
        {
        LOG(0, "guiding pulse limited to 9.999 seconds");
        nTimeMsecs = 9999;
        }
    QString qsDirection;
    switch (eDirection)
        {
        case north: qsDirection = "n"; break;
        case south: qsDirection = "s"; break;
        case east:  qsDirection = "e"; break;
        case west:  qsDirection = "w"; break;
        };
    QString qsCmd = QString::fromUtf8("#:Mg%1%2#")
        .arg(qsDirection).arg(nTimeMsecs, 4, 10, QLatin1Char('0'));
    sendCmd(qsCmd.toLatin1().constData());
    }







void McuCom::cmdQueryObjectCoords()
    {
    char acAns[9];
    sendCmd("<#:Gr#", 8, acAns); acAns[8] = '\0';
    LOG(3, "object RA %s", LOG_QS(QString::fromLatin1(acAns)));
    sendCmd("<#:Gd#", 7, acAns); acAns[7] = '\0';
    // translate degree symbol from LX200 (0xdf) to ASCII (0xb0)
    if (acAns[3] == -33)
        acAns[3] = -80;
    LOG(3, "object Dec %s", LOG_QS(QString::fromLatin1(acAns)));
    }


bool McuCom::cmdSetObjectCoords(int nRaHour, int nRaMin, int nRaMinTenth,
    int nDecSign, int nDecDeg, int nDecMin)
    {
    if ((nRaHour < 0) || (nRaHour > 23) ||
        (nRaMin  < 0) || (nRaMin  > 59) ||
        (nRaMinTenth < 0) || (nRaMinTenth > 9) ||
        (nDecDeg < 0) || (nDecDeg > 90) ||
        (nDecMin < 0) || (nDecMin > 59))
        {
        LOG(0, "wrong coords %02d:%02d.%01d %c%02d°%02d",
            nRaHour, nRaMin, nRaMinTenth,
            (nDecSign < 0)? '-': '+', nDecDeg, nDecMin);
        return false;
        }
    char acAns[1];
    QString qsCmd = QString::fromUtf8("#:Sr%1:%2.%3#")
        .arg(nRaHour, 2, 10, QLatin1Char('0'))
        .arg(nRaMin, 2, 10, QLatin1Char('0'))
        .arg(nRaMinTenth, 1, 10, QLatin1Char('0'));
    sendCmd(qsCmd.toLatin1().constData(), 1, acAns);
    if (acAns[0] != '1')
        {
        LOG(0, "failed to send RA %02d:%02d.%01d %c%02d°%02d",
            nRaHour, nRaMin, nRaMinTenth,
            (nDecSign < 0)? '-': '+', nDecDeg, nDecMin);
        return false;
        }
    qsCmd = QString::fromUtf8("#:Sd%1%2%3%4#")
        .arg(QLatin1Char((nDecSign < 0)? '-': '+'))
        .arg(nDecDeg, 2, 10, QLatin1Char('0'))
        .arg(QLatin1Char(-33))
        .arg(nDecMin, 2, 10, QLatin1Char('0'));
    sendCmd(qsCmd.toLatin1().constData(), 1, acAns);
    if (acAns[0] != '1')
        {
        LOG(0, "failed to send Dec %02d:%02d.%01d %c%02d°%02d",
            nRaHour, nRaMin, nRaMinTenth,
            (nDecSign < 0)? '-': '+', nDecDeg, nDecMin);
        return false;
        }

    return true;
    }


void McuCom::cmdQueryTelescopeCoords()
    {
    char acAns[9];
    sendCmd("<#:GR#", 8, acAns); acAns[8] = '\0';
    LOG(3, "telescope RA %s", QString::fromLatin1(acAns).toUtf8().constData());
    sendCmd("<#:GD#", 7, acAns); acAns[7] = '\0';
    // translate degree symbol from LX200 (0xdf) to ASCII (0xb0)
    if (acAns[3] == -33)
        acAns[3] = -80;
    LOG(3, "telescope Dec %s", QString::fromLatin1(acAns).toUtf8().constData());
    }


void McuCom::cmdSyncTelescopeToObject()
    {
    char acAns[1];
    sendCmd("#:MS#", 1, acAns);
    }


bool McuCom::cmdSlewTelescopeToObject()
    {
    char acAns[1];
    sendCmd("#:MS#", 1, acAns);
    if (acAns[0] != '0')
        {
        // todo: treat further bytes, e.g. " No Goto#"
        LOG(0, "failed to slew to object");
        return false;
        }
    return true;
    }


#endif

