
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2006-2007 by Wolfgang Hoffmann <woho@woho.de>           *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/


#ifndef ASPECT_MCUCMD_H
#define ASPECT_MCUCMD_H


#include "mcuserial.h"

#include <QByteArray>
#include <QString>


class Cmd
    {
public:
    Cmd() { m_aCmd.clear(); };
    Cmd(const char *szCmd) { m_aCmd.clear(); str(szCmd); };

    Cmd &raw(const QByteArray &aCmd);
    Cmd &str(const char *szCmd);
    Cmd &num(int nData);
    Cmd &byte(int nData);
    Cmd &word(int nData);
    Cmd &real(float fData);

    QByteArray command() const { return m_aCmd; };

private:
    QByteArray m_aCmd;
    };


class Ans
    {
public:
    QByteArray answer() const { return m_aAns; };
    int byte(int nIx) const;
    int word(int nIx) const;
    float real(int nIx) const;

protected:
    QByteArray m_aAns;
    };


class McuCom : public IMcuCmdAns, public Ans
    {
public:
    McuCom(const char *szName, const Cmd &oCmd)
        : m_qsName(QString::fromUtf8(szName)), m_aCmd(oCmd.command()) {};
    bool send(const McuSerial &oPort, int nTimeout = 200);

private:
    QString name() const { return m_qsName; };
    QByteArray command() const { return m_aCmd; };

protected:
    QString m_qsName;
    QByteArray m_aCmd;
    };


class McuAns : public IMcuAns, public Ans
    {
public:
    McuAns(const char *szName, int nAnsMin, int nAnsMax)
        : m_qsName(QString::fromUtf8(szName)),
          m_nAnsMin(nAnsMin), m_nAnsMax(nAnsMax) {};

private:
    void receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment);
    QString name() const { return m_qsName; };

protected:
    QString m_qsName;
    int m_nAnsMin;
    int m_nAnsMax;
    };


class McuCmdVoid : public McuCom
    {
public:
    McuCmdVoid(const char *szName, const Cmd &oCmd)
        : McuCom(szName, oCmd) {};

private:
    void receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment);
    };


class McuCmdStd : public McuCom
    {
public:
    McuCmdStd(const char *szName, const Cmd &oCmd)
        : McuCom(szName, oCmd), m_bSuccess(true) {};
    bool success() const;

private:
    void receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment);

protected:
    bool m_bSuccess;
    };


class McuCmdBool : public McuCom
    {
public:
    McuCmdBool(const char *szName, const Cmd &oCmd, char cSuccess = '0')
        : McuCom(szName, oCmd), m_cSuccess(cSuccess) {};
    bool success() const;
    QString errorReason() const;

private:
    void receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment);

protected:
    char m_cSuccess;
    };


class McuCmdString : public McuCom
    {
public:
    McuCmdString(const char *szName, const Cmd &oCmd)
        : McuCom(szName, oCmd) {};

private:
    void receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment);
    };


class McuCmdFixed : public McuCom
    {
public:
    McuCmdFixed(const char *szName, const Cmd &oCmd, int nAnsLen)
        : McuCom(szName, oCmd), m_nAnsLen(nAnsLen) {};

private:
    void receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment);

protected:
    int m_nAnsLen;
    };


class McuCmdQueryUnlimited : public McuCom
    {
public:
    McuCmdQueryUnlimited(const char *szName, const Cmd &oCmd, int nAnsMin)
        : McuCom(szName, oCmd), m_nAnsMin(nAnsMin) {};

private:
    void receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment);

protected:
    int m_nAnsMin;
    };


#endif // ASPECT_MCUCMD_H

