
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2006-2007 by Wolfgang Hoffmann <woho@woho.de>           *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/


#include "mcucmd.h"
#include "log.h"




/********************************************************************/
/*!
    \class Cmd
    \brief Helper class for building command strings
*/


/*!
    Append raw bytearray \a aCmd to command buffer
*/

Cmd &Cmd::raw(const QByteArray &aCmd)
    {
    m_aCmd.append(aCmd);
    return *this;
    }


/*!
    Append string \a szCmd to command buffer
*/

Cmd &Cmd::str(const char *szCmd)
    {
    m_aCmd.append(QString::fromUtf8(szCmd).toLatin1());
    return *this;
    }


/*!
    Append integer number \a nData as ASCII string to command buffer
*/

Cmd &Cmd::num(int nData)
    {
    m_aCmd.append(QString::fromUtf8("%1").arg(nData).toLatin1());
    return *this;
    }


/*!
    Append byte \a nData as 8 bit binary data to command buffer
*/

Cmd &Cmd::byte(int nData)
    {
    m_aCmd.append((char)nData);
    return *this;
    }


/*!
    Append word \a nData as 16 bit binary data to command buffer
*/

Cmd &Cmd::word(int nData)
    {
    m_aCmd.append((char)(nData & 0xff));
    m_aCmd.append((char)((nData >> 8) & 0xff));
    return *this;
    }


/*!
    Append float \a fData as 32 bit binary IEEE float to command buffer
*/

Cmd &Cmd::real(float fData)
    {
    m_aCmd.append(((char *)&fData)[0]);
    m_aCmd.append(((char *)&fData)[1]);
    m_aCmd.append(((char *)&fData)[2]);
    m_aCmd.append(((char *)&fData)[3]);
    return *this;
    }




/********************************************************************/
/*!
    \class Ans
    \brief Helper class for parsing answer strings
*/


/*!
    Return value of binary answer byte (8 bit) at offset \a nIx
*/

int Ans::byte(int nIx) const
    {
    if ((nIx < 0) || (nIx >= m_aAns.size()))
        return 0;
    return (int)(unsigned char)m_aAns[nIx];
    }


/*!
    Return value of binary answer word (16 bit, little endian) at offset \a nIx
*/

int Ans::word(int nIx) const
    {
    if ((nIx < 0) || (nIx+1 >= m_aAns.size()))
        return 0;
    return (int)(unsigned char)m_aAns[nIx] +
        (((int)(unsigned char)m_aAns[nIx+1]) << 8);
    }


/*!
    Return value of binary answer float (32 bit IEEE float) at offset \a nIx
*/

float Ans::real(int nIx) const
    {
    if ((nIx < 0) || (nIx+3 >= m_aAns.size()))
        return 0.;
    // copy binary image of 4 bytes into float
    float fAns = 0.;
    ((char *)&fAns)[0] = m_aAns[nIx];
    ((char *)&fAns)[1] = m_aAns[nIx+1];
    ((char *)&fAns)[2] = m_aAns[nIx+2];
    ((char *)&fAns)[3] = m_aAns[nIx+3];
    return fAns;
    }




/********************************************************************/
/*!
    \class McuCom
    \brief Generic base for MCU communication (command and opt. answer)
*/

bool McuCom::send(const McuSerial &oPort, int nTimeout)
    {
    McuSerial *pPort = const_cast<McuSerial *>(&oPort);
    pPort->flush();
    m_aAns.clear();
    return (pPort->send(this) && pPort->recv(this, nTimeout));
    }




/********************************************************************/
/*!
    \class McuAns
    \brief Generic base for MCU communication (answer only)
*/

void McuAns::receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment)
    {
    m_aAns += aAnswerFragment;
    rnMin = m_nAnsMin - m_aAns.length();
    rnMax = m_nAnsMax - m_aAns.length();
    }




/********************************************************************/
/*!
    \class McuComVoid
    \brief Generic base for commands with no answer
*/

void McuCmdVoid::receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment)
    {
    Q_UNUSED(aAnswerFragment)
    rnMin = 0;
    rnMax = 0;
    }




/********************************************************************/
/*!
    \class McuCmdStd
    \brief Generic base for commands with no answer, but '?' error indicator
*/

void McuCmdStd::receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment)
    {
    // a standard LittleFoot command gives no answer,
    // unless an error occured; then '?' is returned
    // wait for 50 msecs if an answer came, and if not, assume success.
    if (aAnswerFragment.isEmpty())
        McuSerial::msleep(50);
    else
        {
        m_aAns += aAnswerFragment;
        m_bSuccess = false;
        }
    rnMin = 0;
    rnMax = 1000;
    }




/********************************************************************/
/*!
    \class McuCmdBool
    \brief Generic base for commands with boolean answer and optional error string
*/

bool McuCmdBool::success() const
    {
    return (m_aAns.length() > 0) && (m_aAns[0] == m_cSuccess);
    }


QString McuCmdBool::errorReason() const
    {
    if ((m_aAns.length() < 1) || (m_aAns[0] == m_cSuccess))
        return QString();
    return QString::fromLatin1(m_aAns.mid(1));
    }


void McuCmdBool::receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment)
    {
    // wait for at least 1 answer character until
    // status character is available and is m_cSuccess, or
    // status character and error string delimited by # is avaiable
    m_aAns += aAnswerFragment;
    rnMin = 1;
    rnMax = 1000;
    if (((m_aAns.length() > 0) && (m_aAns[0] == m_cSuccess)) ||
        ((m_aAns.length() > 1) && (m_aAns.endsWith("#"))))
        rnMax = 0;
    }




/********************************************************************/
/*!
    \class McuCmdString
    \brief Generic base for commands with string answer (delimited by #)
*/

void McuCmdString::receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment)
    {
    // wait for at least 1 answer character
    // after that, keep receiving if string not delimited by #
    // until no further data is available
    m_aAns += aAnswerFragment;
    rnMin = 1;
    rnMax = 1000;
    if (m_aAns.endsWith("#"))
        rnMax = 0;
    }




/********************************************************************/
/*!
    \class McuCmdFixed
    \brief Generic base for commands with fixed lenght answer
*/

void McuCmdFixed::receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment)
    {
    // wait for exactly m_nAnsLen answer characters
    m_aAns += aAnswerFragment;
    rnMin = m_nAnsLen - m_aAns.length();
    rnMax = rnMin;
    }




/********************************************************************/
/*!
    \class McuCmdQueryUnlimited
    \brief Generic base for querying undelimited strings
*/

void McuCmdQueryUnlimited::receive(int &rnMin, int &rnMax, const QByteArray &aAnswerFragment)
    {
    // wait at least 50 msecs, and receive at least m_nAnsMin bytes.
    // after that, take what's available,
    if (aAnswerFragment.isEmpty())
        McuSerial::msleep(50);
    else
        m_aAns += aAnswerFragment;
    rnMin = m_nAnsMin - m_aAns.length();
    rnMax = 1000;
    }



