
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2006-2007 by Wolfgang Hoffmann <woho@woho.de>           *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/


#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "data.h"
#include "log.h"
#include "settings.h"
#include "mount.h"
#include "logwindow.h"
#include "mcuproxy.h"

#include <QtGui>
#include <math.h>

/*
#define DEBUG_WOHO
*/




/********************************************************************/
/*!
    \class Progress
    \internal
*/

class Progress : public IProgress
    {
public:
    Progress(QProgressDialog &rDialog)
        : m_pDialog(&rDialog)
        {
        };
    void setText(const QString &qsText)
        {
        m_pDialog->setLabelText(qsText);
        };
    bool setValue(double dfValue)
        {
        m_pDialog->setValue((int)((double)m_pDialog->maximum() * dfValue));
        qApp->processEvents();
        return m_pDialog->wasCanceled();
        };
private:
    QProgressDialog *m_pDialog;
    };




/********************************************************************/
/*!
    \class StatusBar
    \internal
*/

class StatusBar
    {
public:
    enum Led { clear_off, red_off, red_on, yellow_off, yellow_on, green_off, green_on };

    StatusBar(QStatusBar *pStatusBar);
    void setMcuPort(const QString &qsPort);
    void setMcuStatus(StatusBar::Led eLed);
    void setFirmware(const QString &qsFirmware);
    void setPecStatus(StatusBar::Led eLed);

private:
    QPixmap led(Led eLed);

    QStatusBar *m_pStatusBar;
    QLabel *m_pMcuPort;
    QLabel *m_pMcuStatus;
    QLabel *m_pFirmware;
    QLabel *m_pPecStatus;
    };


StatusBar::StatusBar(QStatusBar *pStatusBar)
:   m_pStatusBar(pStatusBar)
    {
    QWidget *pWidget = new QWidget(pStatusBar);
    QHBoxLayout *pLayout = new QHBoxLayout(pWidget);
    pLayout->setMargin(2);
    m_pMcuPort = new QLabel(pWidget);
    pLayout->addWidget(m_pMcuPort);
    m_pMcuStatus = new QLabel(pWidget);
    pLayout->addWidget(m_pMcuStatus);
    //m_pStatusBar->addPermanentWidget(pWidget);
    m_pStatusBar->addWidget(pWidget);

    pWidget = new QWidget(pStatusBar);
    pLayout = new QHBoxLayout(pWidget);
    pLayout->setMargin(2);
    m_pFirmware = new QLabel(pWidget);
    m_pFirmware->setMinimumWidth(m_pFirmware->fontMetrics().width(QString::fromUtf8("v4.00 Rev.D")));
    pLayout->addWidget(m_pFirmware);
    m_pStatusBar->addWidget(pWidget);

    pWidget = new QWidget(pStatusBar);
    pLayout = new QHBoxLayout(pWidget);
    pLayout->setMargin(2);
    m_pPecStatus = new QLabel(pWidget);
    pLayout->addWidget(m_pPecStatus);
#if 0
    m_pStatusBar->addPermanentWidget(pWidget);
    m_pStatusBar->addWidget(pWidget);
#endif
    }


void StatusBar::setMcuPort(const QString &qsPort)
    {
    m_pMcuPort->setText(QString::fromUtf8("%1").arg(qsPort));
    }


void StatusBar::setMcuStatus(StatusBar::Led eLed)
    {
    m_pMcuStatus->setPixmap(led(eLed));
    }


void StatusBar::setFirmware(const QString &qsFirmware)
    {
    m_pFirmware->setText(qsFirmware);
    }


void StatusBar::setPecStatus(StatusBar::Led eLed)
    {
    m_pPecStatus->setPixmap(led(eLed));
    }


QPixmap StatusBar::led(StatusBar::Led eLed)
    {
    switch(eLed)
        {
        case clear_off:
            return QIcon(":/icons/led_clear_off.png").pixmap(QSize(16, 16));
        case red_off:
            return QIcon(":/icons/led_red_off.png").pixmap(QSize(16, 16));
        case red_on:
            return QIcon(":/icons/led_red_on.png").pixmap(QSize(16, 16));
        case yellow_off:
            return QIcon(":/icons/led_yellow_off.png").pixmap(QSize(16, 16));
        case yellow_on:
            return QIcon(":/icons/led_yellow_on.png").pixmap(QSize(16, 16));
        case green_off:
            return QIcon(":/icons/led_green_off.png").pixmap(QSize(16, 16));
        case green_on:
            return QIcon(":/icons/led_green_on.png").pixmap(QSize(16, 16));
        }
    return QPixmap(QSize(16, 16));
    }




/********************************************************************/
/*!
    \class MainWindow
    \brief Unsorted heap of all kind of stuff

    MainWindow is the GUI main window.

    \todo sort out stuff that is not related to main window
        (pe and pec handling and calculations, graphs, ...)
*/

MainWindow::MainWindow()
:   QMainWindow()
    {
    m_pUi = new Ui::MainWindow;
    m_pUi->setupUi(this);

    m_pStatusBar = new StatusBar(m_pUi->statusbar);

    m_pLog = new LogDialog;
    m_pSettings = new SettingsDialog(this);
    m_pMount = new MountDialog(this);

    m_pPeRaw = 0;
    m_aPeProcessed.resize(PecData::c_nRowNum);
    m_aPecIdeal.resize(PecData::c_nRowNum);

    m_pPeProcessedModel = new QStandardItemModel(this);
    m_pUi->viewPeProcessed->setModel(m_pPeProcessedModel);
    m_pPecModel = new QStandardItemModel(this);
    m_pUi->viewPec->setModel(m_pPecModel);
    connect(m_pPecModel, SIGNAL(itemChanged(QStandardItem *)), this, SLOT(changePecItem(QStandardItem *)));
    m_pPecProfileModel = new QStandardItemModel(this);
    m_pUi->viewPecProfile->setModel(m_pPecProfileModel);
    m_pMcuTimer = new QTimer(this);
    connect(m_pMcuTimer, SIGNAL(timeout()), this, SLOT(mcuTimer()));

    m_pMcuProxy = &(McuProxy::proxy());
    connect(m_pMcuProxy, SIGNAL(status(bool, McuSerial::Status)),
        this, SLOT(mcuSerialStatus(bool, McuSerial::Status)));
    connect(m_pMcuProxy, SIGNAL(firmware(QString)),
        this, SLOT(mcuSerialFirmware(const QString &)));
    connect(m_pMcuProxy, SIGNAL(busy(bool)),
        this, SLOT(mcuSerialBusy(bool)));

    m_pPeChart = new QGraphicsScene(this);
    m_pUi->chartPe->setScene(m_pPeChart);

    // Insert edited item into list when combobox looses focus.
    // Qt should do this ...
    if (m_pUi->mcuPort->lineEdit() != 0)
        connect(m_pUi->mcuPort->lineEdit(), SIGNAL(editingFinished()),
            m_pUi->mcuPort->lineEdit(), SIGNAL(returnPressed()));

    m_dfSiderealDay = 86164.091;

    m_eGuideDir = stop;
    m_pUi->mcuPause->setEnabled(false);

    // todo: implement functionality
    m_pUi->pecVerifyTable->hide();
    m_pUi->actionLog_View->setEnabled(false);

/*
    m_pUi->tabPecProfile->setEnabled(false);
    m_pUi->toggleChartPecProfile->setChecked(false);
    m_pUi->toggleChartPecProfile->setEnabled(false);
*/

    // to this after all other initialisations, as it might get
    // deep into code already ...
    m_qsSerialPort = QString();
    m_nWormWheelTeeth = 0;
    Settings::connect(this, SLOT(settingsChanged()));
    settingsChanged();

    resize(960, 640);
    }


MainWindow::~MainWindow()
    {
    m_pMcuTimer->stop();
    delete m_pLog;
    delete m_pUi;
    }


void MainWindow::on_actionQuit_triggered()
    {
    LOG(4, "MainWindow::on_actionQuit_triggered");
    close();
    }


void MainWindow::on_actionAbout_Aspect_triggered()
    {
    LOG(4, "MainWindow::on_actionAbout_Aspect_triggered");
    QMessageBox::about(this,
        tr("About Aspect ..."),
        tr("<h3>Aspect</h3>"
           "<p>Aspect is a simple PEC tool, allowing to "
           "view, edit and calculate periodic error correction "
           "tables for the <a href=\"http://littlefoot.rajiva.de/\">LittleFoot</a> "
           "astronomical mount stepper controler.</p>"
           "<p>Version: <b>0.8.1</b> (24 Mar 2007)<br>"
           "Autor: Wolfgang Hoffmann (<a href=\"mailto:woho@woho.de\">woho@woho.de</a>)<br>"
           "Homepage: <a href=\"http://www.woho.de/devel/astro/aspect.html\">"
           "http://www.woho.de/devel/astro/aspect.html</a></p>"
           "<p>Copyright (C) 2006-2007 Wolfgang Hoffmann.<br>"
           "This program is free software, licensed under the GPL v2.</p>"
           "<p>This program is distributed in the hope that it will be useful, "
           "but WITHOUT ANY WARRANTY; without even the implied warranty of "
           "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the "
           "GNU General Public License for more details.</p>"));
    }


void MainWindow::on_actionAbout_Qt_triggered()
    {
    LOG(4, "MainWindow::on_actionAbout_Qt_triggered");
    QApplication::aboutQt();
    }


void MainWindow::on_actionLog_View_triggered()
    {
    m_pLog->show();
    }


void MainWindow::on_actionSettings_triggered()
    {
    m_pSettings->show();
    }


void MainWindow::on_actionMount_triggered()
    {
    m_pMount->show();
    }




/********************************************************************/
/*
    Settings
*/

void MainWindow::settingsChanged()
    {
    if (m_qsSerialPort != Settings::getMcuPort())
        {
        m_qsSerialPort = Settings::getMcuPort();
        int nIndex = m_pUi->mcuPort->findText(m_qsSerialPort);
        if (nIndex == -1)
            {
            nIndex = m_pUi->mcuPort->count();
            m_pUi->mcuPort->insertItem(nIndex, m_qsSerialPort);
            }
        m_pUi->mcuPort->setCurrentIndex(nIndex);
        m_pStatusBar->setMcuPort(m_qsSerialPort);
        m_pStatusBar->setMcuStatus(StatusBar::clear_off);
        }
    calcPecParams();
    }


/********************************************************************/
/*
    MCU connection
*/


void MainWindow::on_mcuPort_activated(const QString &qsPort)
    {
    Settings::setMcuPort(qsPort);
    }


void MainWindow::mcuSerialStatus(bool bOpen, McuSerial::Status eStatus)
    {
    LOG(4, "mcuSerialStatus(%s, %s)", bOpen? "open": "closed",
        (eStatus == McuSerial::unknown)? "unknown":
        (eStatus == McuSerial::error)? "error":
        (eStatus == McuSerial::cmd_ok)? "cmd_ok":
        (eStatus == McuSerial::ans_ok)? "ans_ok": "???");
    switch (eStatus)
        {
        case McuSerial::unknown:
            m_pStatusBar->setMcuStatus(StatusBar::clear_off);
            break;
        case McuSerial::error:
            if (bOpen)
                m_pStatusBar->setMcuStatus(StatusBar::red_on);
            else
                m_pStatusBar->setMcuStatus(StatusBar::red_off);
            break;
        case McuSerial::cmd_ok:
            if (bOpen)
                m_pStatusBar->setMcuStatus(StatusBar::yellow_on);
            else
                m_pStatusBar->setMcuStatus(StatusBar::yellow_off);
            break;
        case McuSerial::ans_ok:
            if (bOpen)
                m_pStatusBar->setMcuStatus(StatusBar::green_on);
            else
                m_pStatusBar->setMcuStatus(StatusBar::green_off);
            break;
        }
    }


void MainWindow::mcuSerialFirmware(const QString &qsFirmware)
    {
    m_pStatusBar->setFirmware(qsFirmware);
    }


void MainWindow::mcuSerialBusy(bool bBusy)
    {
    if (bBusy)
        {
        m_pUi->mcuSettingsTracking->setEnabled(false);
        m_pUi->mcuSettingsPec->setEnabled(false);
        m_pUi->mcuSettingsIntelly->setEnabled(false);
        m_pUi->mcuSettingsRefresh->setEnabled(false);
        m_pUi->pecGetTable->setEnabled(false);
        m_pUi->pecPutTable->setEnabled(false);
        m_pUi->pecVerifyTable->setEnabled(false);
        m_pUi->pecMountQuery->setEnabled(false);
        m_pUi->mcuGuideEast->setEnabled(false);
        m_pUi->mcuGuideWest->setEnabled(false);
        if (!m_pUi->mcuStart->isChecked())
            m_pUi->mcuStart->setEnabled(false);
        if (m_pUi->mcuPause->isChecked())
            m_pUi->mcuPause->setEnabled(false);
        }
    else
        {
        m_pUi->mcuSettingsTracking->setEnabled(true);
        m_pUi->mcuSettingsPec->setEnabled(true);
        m_pUi->mcuSettingsIntelly->setEnabled(true);
        m_pUi->mcuSettingsRefresh->setEnabled(true);
        m_pUi->pecGetTable->setEnabled(true);
        m_pUi->pecPutTable->setEnabled(true);
        m_pUi->pecVerifyTable->setEnabled(true);
        m_pUi->pecMountQuery->setEnabled(true);
        m_pUi->mcuGuideEast->setEnabled(true);
        m_pUi->mcuGuideWest->setEnabled(true);
        if (!m_pUi->mcuStart->isChecked())
            m_pUi->mcuStart->setEnabled(true);
        if (m_pUi->mcuPause->isChecked())
            m_pUi->mcuPause->setEnabled(true);
        }
    }


void MainWindow::on_mcuSettingsTracking_activated(int nSpeed)
    {
    bool bSuccess = false;
    if (mcuCommunicationBegin())
        {
        bSuccess = m_pMcuProxy->setTrackingRate(
            (nSpeed == 0)? McuProxy::sidereal:
            (nSpeed == 1)? McuProxy::solar:
            (nSpeed == 2)? McuProxy::lunar: McuProxy::off);
        mcuCommunicationEnd();
        }
    if (!bSuccess)
        {
        m_pUi->mcuSettingsTracking->setCurrentIndex(-1);
        QMessageBox::warning(this,
            tr("Aspect - MCU Communication"),
            tr("Failed to set tracking rate.\n\n"
               "Please check that MCU is switched on and properly connected,\n"
               "and that the correct serial port is selected."),
            QMessageBox::Ok, QMessageBox::NoButton);
        }
    }


void MainWindow::on_mcuSettingsPec_activated(int nIndex)
    {
    if (m_pUi->mcuSettingsPec->count() == 1)
        return;
    bool bSuccess = false;
    if (mcuCommunicationBegin())
        {
        bSuccess = m_pMcuProxy->setPec(nIndex == 1);
        mcuCommunicationEnd();
        }
    if (!bSuccess)
        {
        m_pUi->mcuSettingsPec->setCurrentIndex(-1);
        QMessageBox::warning(this,
            tr("Aspect - MCU Communication"),
            tr("Failed to switch PEC on/off.\n\n"
               "Please check that MCU is switched on and properly connected,\n"
               "and that the correct serial port is selected."),
            QMessageBox::Ok, QMessageBox::NoButton);
        }
    }


void MainWindow::on_mcuSettingsIntelly_activated(int nIndex)
    {
    bool bSuccess = false;
    if (mcuCommunicationBegin())
        {
        // todo: check if IntellyTrack-key is licensed before switching on
        bSuccess = m_pMcuProxy->setIntellyTrack(nIndex == 1);
        mcuCommunicationEnd();
        }
    if (!bSuccess)
        {
        m_pUi->mcuSettingsPec->setCurrentIndex(-1);
        QMessageBox::warning(this,
            tr("Aspect - MCU Communication"),
            tr("Failed to switch IntellyTrack on/off.\n\n"
               "Please check that MCU is switched on and properly connected,\n"
               "and that the correct serial port is selected."),
            QMessageBox::Ok, QMessageBox::NoButton);
        }
    }


void MainWindow::on_mcuSettingsRefresh_clicked()
    {
    McuProxy::Tracking eTracking = McuProxy::off;
    McuDataPecState oPec;
    bool bIntelly = false;
    bool bSuccess = false;
    if (mcuCommunicationBegin())
        {
        eTracking = m_pMcuProxy->getTrackingRate();
        oPec = m_pMcuProxy->getPecState();
        bIntelly = m_pMcuProxy->getIntellyTrack();
        bSuccess = true;
        mcuCommunicationEnd();
        }
    // in case of error, first update GUI before opening message box

    switch (eTracking)
        {
        case McuProxy::sidereal: m_pUi->mcuSettingsTracking->setCurrentIndex(0); break;
        case McuProxy::solar:    m_pUi->mcuSettingsTracking->setCurrentIndex(1); break;
        case McuProxy::lunar:    m_pUi->mcuSettingsTracking->setCurrentIndex(2); break;
        case McuProxy::off:      m_pUi->mcuSettingsTracking->setCurrentIndex(3); break;
        }

    // dynamically change contents of PEC dropdown, as single entries cannot be disabled
    if (oPec.pec() < 2)
        {
        m_pUi->mcuSettingsPec->setItemText(0, tr("Off"));
        if (m_pUi->mcuSettingsPec->count() == 1)
            m_pUi->mcuSettingsPec->addItem(tr("On"));
        else
            m_pUi->mcuSettingsPec->setItemText(1, tr("On"));
        m_pUi->mcuSettingsPec->setCurrentIndex(oPec.pec());
        }
    else
        {
        if (m_pUi->mcuSettingsPec->count() == 2)
            m_pUi->mcuSettingsPec->removeItem(1);
        m_pUi->mcuSettingsPec->setCurrentIndex(0);
        if (oPec.pec() == 2)
            m_pUi->mcuSettingsPec->setItemText(0, tr("On by Handbox"));
        else if (oPec.pec() == 3)
            m_pUi->mcuSettingsPec->setItemText(0, tr("In Training"));
        else if (oPec.pec() == 4)
            m_pUi->mcuSettingsPec->setItemText(0, tr("Off (State not OK)"));
        else
            m_pUi->mcuSettingsPec->setItemText(0, tr(""));
        }

    m_pUi->mcuSettingsIntelly->setCurrentIndex((bIntelly)? 1: 0);

    if (bSuccess)
        {
        LOG(1, "querying MCU status: Tracking %s, PEC %s, Intelly %s",
            (eTracking == McuProxy::sidereal)? "sidereal":
            (eTracking == McuProxy::solar)? "solar":
            (eTracking == McuProxy::lunar)? "lunar":
            (eTracking == McuProxy::off)? "off": "???",
            (oPec.pec() == 0)? "off":
            (oPec.pec() == 1)? "on (serial)":
            (oPec.pec() == 2)? "on (handbox)":
            (oPec.pec() == 3)? "on (training)":
            (oPec.pec() == 4)? "off (not ok)": "???",
            (bIntelly)? "on": "off");
        }
    else
        {
        m_pUi->mcuSettingsTracking->setCurrentIndex(-1);
        m_pUi->mcuSettingsPec->setCurrentIndex(-1);
        m_pUi->mcuSettingsIntelly->setCurrentIndex(-1);
        QMessageBox::warning(this,
            tr("Aspect - MCU Communication"),
            tr("Failed to query settings from MCU.\n\n"
               "Please check that MCU is switched on and properly connected,\n"
               "and that the correct serial port is selected."),
            QMessageBox::Ok, QMessageBox::NoButton);
        }
    }


bool MainWindow::mcuCommunicationBegin()
    {
    QApplication::setOverrideCursor(Qt::BusyCursor);
    bool bSuccess = m_pMcuProxy->open(m_pUi->mcuPort->currentText());
    QApplication::restoreOverrideCursor();
    return bSuccess;
    }


void MainWindow::mcuCommunicationEnd()
    {
    m_pMcuProxy->close();
    }




/********************************************************************/
/*
    PE raw data
*/


void MainWindow::on_actionOpen_PE_Data_triggered()
    {
    LOG(4, "MainWindow::on_actionOpen_PE_Data_triggered");

    // switch off polling
    m_pUi->peFileFollow->setChecked(false);

    QFileDialog oFileDialog(this);
    oFileDialog.setAcceptMode(QFileDialog::AcceptOpen);
    oFileDialog.setFileMode(QFileDialog::ExistingFile);
    oFileDialog.setViewMode(QFileDialog::Detail);
    oFileDialog.setWindowTitle(tr("Open PE Data File"));
    oFileDialog.setHistory(Settings::getPathHistory());
    oFileDialog.setDirectory(Settings::getPathDefault());
    QStringList qslFilters;
    qslFilters << tr("MCU Control Plus Format (*.csv)");
    qslFilters << tr("IRIS Format (*.dat)");
    qslFilters << tr("All Files (*)");
    oFileDialog.setFilters(qslFilters);

    QStringList qslFileNames;
    if (oFileDialog.exec())
        {
        Settings::setPathDefault(oFileDialog.directory().absolutePath());
        qslFileNames = oFileDialog.selectedFiles();
        }
    if (qslFileNames.size() > 0)
        {
        QString qsFileName = qslFileNames.at(0);
        if (!openPeFile(qsFileName))
            {
            QMessageBox::warning(this,
                tr("Aspect - Load PE File"),
                tr("Failed to load PE measurement data from file\n\n%1\n\n"
                   "Make sure that the selected file is a PE log file\n"
                   "measured with MCU Control Plus (version 2.1.7 or greater)")
                    .arg(qsFileName),
                QMessageBox::Ok, QMessageBox::NoButton);
            }
        }
    }


void MainWindow::on_peFileFollow_clicked()
    {
    LOG(0, "MainWindow::on_peFileFollow_clicked");
    if (m_pPeRaw == 0)
        return;
    if (!m_pPeRaw->pollPeFile(m_pUi->peFileFollow->isChecked()))
        m_pUi->peFileFollow->setChecked(false);
    }


bool MainWindow::openPeFile(const QString &qsFileName)
    {
    LOG(4, "MainWindow::openPeFile");
    LOG(1, "loading PE data \"%s\"", qsFileName.toLocal8Bit().constData());
    m_pUi->lineEditPeFile->setText(qsFileName);

    if (m_pPeRaw != 0)
        m_pPeRaw->clearData();

    if (m_pUi->widgetPeMcu->probePeFile(qsFileName))
        {
        m_pUi->stackPeFile->setCurrentWidget(m_pUi->pagePeMcu);
        m_pPeRaw = m_pUi->widgetPeMcu;
        }
    else if (m_pUi->widgetPeIris->probePeFile(qsFileName))
        {
        m_pUi->stackPeFile->setCurrentWidget(m_pUi->pagePeIris);
        m_pPeRaw = m_pUi->widgetPeIris;
        }
    else
        {
        m_pUi->stackPeFile->setCurrentWidget(m_pUi->pagePeNone);
        m_pPeRaw = 0;
        }

    if (m_pPeRaw != 0)
        {
        // will emit dataPeChanged() connected to recalcPeData()
        m_pPeRaw->setPeFileName(qsFileName);
        if (m_pPeRaw->loadPeFile())
            m_pUi->peFileFollow->setEnabled(true);
        else
            {
            m_pUi->stackPeFile->setCurrentWidget(m_pUi->pagePeNone);
            m_pPeRaw = 0;
            }
        }
    if (m_pPeRaw == 0)
        {
        LOG(1, "    ... failed.");
        m_pUi->lineEditPeFile->setText(QString());
        m_pUi->peFileFollow->setEnabled(false);
        m_pUi->peFileFollow->setChecked(false);
        recalcPeData();
        }
    return (m_pPeRaw != 0);
    }


void MainWindow::on_widgetPeMcu_dataPeChanged(IPeRawData *pData)
    {
    LOG(4, "MainWindow::on_widgetPeMcu_dataPeChanged");
    if (m_pPeRaw == pData)
        recalcPeData();
    }


void MainWindow::on_widgetPeIris_dataPeChanged(IPeRawData *pData)
    {
    LOG(4, "MainWindow::on_widgetPeIris_dataPeChanged");
    if (m_pPeRaw == pData)
        recalcPeData();
    }




/********************************************************************/
/*
    PE processing
*/


void MainWindow::on_peProcMode_activated(int nMode)
    {
    LOG(4, "MainWindow::on_peProcMode_activated");
    Q_UNUSED(nMode)
    recalcPeData();
    }


void MainWindow::on_peProcBeg_valueChanged(int nBeg)
    {
    LOG(4, "MainWindow::on_peProcBeg_valueChanged");
    Q_UNUSED(nBeg)
    recalcPeData();
    }


void MainWindow::on_peProcEnd_valueChanged(int nEnd)
    {
    LOG(4, "MainWindow::on_peProcEnd_valueChanged");
    Q_UNUSED(nEnd)
    recalcPeData();
    }


void MainWindow::on_peProcCombining_activated(int nMethod)
    {
    LOG(4, "MainWindow::on_peProcCombining_activated");
    Q_UNUSED(nMethod)
    recalcPeData();
    }


void MainWindow::on_peProcSmoothWidth_valueChanged(int nWidth)
    {
    LOG(4, "MainWindow::on_peProcSmoothWidth_valueChanged");
    Q_UNUSED(nWidth)
    recalcPeData();
    }


void MainWindow::on_peProcDrift_valueChanged(double dfDrift)
    {
    LOG(4, "MainWindow::on_peProcDrift_valueChanged");
    Q_UNUSED(dfDrift)
    recalcPeData();
    }


void MainWindow::on_peProcDecl_valueChanged(double dfDeclination)
    {
    LOG(4, "MainWindow::on_peProcDecl_valueChanged");
    Q_UNUSED(dfDeclination)
    recalcPeData();
    }


void MainWindow::on_peProcScale_valueChanged(double dfScale)
    {
    LOG(4, "MainWindow::on_peProcScale_valueChanged");
    Q_UNUSED(dfScale)
    recalcPeData();
    }


void MainWindow::on_peProcOffset_valueChanged(double dfOffset)
    {
    LOG(4, "MainWindow::on_peProcOffset_valueChanged");
    Q_UNUSED(dfOffset)
    recalcPeData();
    }


void MainWindow::on_toggleChartPeRaw_toggled(bool bOn)
    {
    LOG(4, "MainWindow::on_toggleChartPeRaw_toggled");
    Q_UNUSED(bOn)
    recalcPeChart();
    }


void MainWindow::on_toggleChartPeProcessed_toggled(bool bOn)
    {
    LOG(4, "MainWindow::on_toggleChartPeProcessed_toggled");
    Q_UNUSED(bOn)
    recalcPeChart();
    }


void MainWindow::on_toggleChartPecIdeal_toggled(bool bOn)
    {
    LOG(4, "MainWindow::on_toggleChartPecIdeal_toggled");
    Q_UNUSED(bOn)
    recalcPeChart();
    }


void MainWindow::on_toggleChartPecTable_toggled(bool bOn)
    {
    LOG(4, "MainWindow::on_toggleChartPecTable_toggled");
    Q_UNUSED(bOn)
    recalcPeChart();
    }


void MainWindow::on_toggleChartPecProfile_toggled(bool bOn)
    {
    LOG(4, "MainWindow::on_toggleChartPecProfile_toggled");
    Q_UNUSED(bOn)
    recalcPeChart();
    }




/********************************************************************/
/*
    PEC table
*/


void MainWindow::on_pecOpenTable_clicked()
    {
    LOG(4, "MainWindow::on_pecOpenTable_clicked");
    QFileDialog oFileDialog(this);
    oFileDialog.setAcceptMode(QFileDialog::AcceptOpen);
    oFileDialog.setFileMode(QFileDialog::ExistingFile);
    oFileDialog.setViewMode(QFileDialog::Detail);
    oFileDialog.setWindowTitle(tr("Open PEC Table File"));
    oFileDialog.setHistory(Settings::getPathHistory());
    oFileDialog.setDirectory(Settings::getPathDefault());
    QStringList qslFilters;
    qslFilters << tr("PEC File (*.csv)");
    qslFilters << tr("All Files (*)");
    oFileDialog.setFilters(qslFilters);

    QStringList qslFileNames;
    if (oFileDialog.exec())
        {
        Settings::setPathDefault(oFileDialog.directory().absolutePath());
        qslFileNames = oFileDialog.selectedFiles();
        }
    if (qslFileNames.size() > 0)
        {
        QString qsFileName = qslFileNames.at(0);
        if (!openPecFile(qsFileName))
            {
            QMessageBox::warning(this,
                tr("Aspect - Load PEC Table File"),
                tr("Failed to load PEC table from file\n\n%1")
                    .arg(qsFileName),
                QMessageBox::Ok, QMessageBox::NoButton);
            }
        }
    }


bool MainWindow::openPecFile(const QString &qsFileName)
    {
    LOG(4, "MainWindow::openPecFile");
    LOG(1, "loading PEC table \"%s\"", qsFileName.toLocal8Bit().constData());

    QFile oFile(qsFileName);
    if (!oFile.open(QIODevice::ReadOnly | QIODevice::Text))
        {
        LOG(1, "    ... failed: cannot open file.");
        return false;
        }
    QTextStream oStream(&oFile);
    QString qsHeader = oStream.readLine();
    if (qsHeader != QString::fromUtf8("Factor, Table, ReqSpeed, IsSpeed, MinSpeed, MaxSpeed"))
        {
        LOG(1, "    ... failed: wrong file format.");
        return false;
        }
    m_oPecTable.clear();
    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        {
        if (oStream.atEnd())
            {
            LOG(1, "    ... failed: unexpected end of file.");
            return false;
            }
        QString qsLine = oStream.readLine();
        LOG(5, "    %s", qsLine.toLocal8Bit().constData());
        QStringList qslEntryStrings = qsLine.split(",");
        int nFactor = (qslEntryStrings.size() > 0)? qslEntryStrings.at(0).toInt(): 1;
        int nTable = (qslEntryStrings.size() > 1)? qslEntryStrings.at(1).toInt(): -1;
        /* these are obsolete with firmware v3.59 and upwards
           (probably even earlier firmware versions)
        int nReqSpeed = (qslEntryStrings.size() > 2)? qslEntryStrings.at(2).toInt(): -1;
        int nIsSpeed = (qslEntryStrings.size() > 3)? qslEntryStrings.at(3).toInt(): -1;
        int nMinSpeed = (qslEntryStrings.size() > 4)? qslEntryStrings.at(4).toInt(): -1;
        int nMaxSpeed = (qslEntryStrings.size() > 5)? qslEntryStrings.at(5).toInt(): -1;
        */
        if (nRow == 0)
            m_oPecTable.setFileFactorAggress(nFactor);
        m_oPecTable.setValue(nRow, PecSample::int2Val(nTable));
        }
    m_oPecTable.setFinish();
    oFile.close();

    LOG(1, "    factor %d, aggressiveness %d",
        m_oPecTable.factor(), m_oPecTable.aggress());
    QString qsTable;
    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        {
        qsTable += QString::fromUtf8("%1").arg(m_oPecTable.valueInt(nRow));
        if (((nRow % 32) == 32 - 1) || (nRow == PecData::c_nRowNum - 1))
            {
            LOG(1, "        %s", LQS(qsTable));
            qsTable.clear();
            }
        }

    fillPecModel();
    recalcPeChart();
    return true;
    }


void MainWindow::on_pecSaveTable_clicked()
    {
    LOG(4, "MainWindow::on_pecSaveTable_clicked");
    QFileDialog oFileDialog(this);
    oFileDialog.setAcceptMode(QFileDialog::AcceptSave);
    oFileDialog.setFileMode(QFileDialog::AnyFile);
    oFileDialog.setViewMode(QFileDialog::Detail);
    oFileDialog.setWindowTitle(tr("Save PEC Table As File"));
    oFileDialog.setHistory(Settings::getPathHistory());
    oFileDialog.setDirectory(Settings::getPathDefault());
    QStringList qslFilters;
    qslFilters << tr("PEC File (*.csv)");
    qslFilters << tr("All Files (*)");
    oFileDialog.setFilters(qslFilters);
    oFileDialog.setDefaultSuffix(QString::fromUtf8("csv"));

    QStringList qslFileNames;
    if (oFileDialog.exec())
        {
        Settings::setPathDefault(oFileDialog.directory().absolutePath());
        qslFileNames = oFileDialog.selectedFiles();
        }
    if (qslFileNames.size() > 0)
        {
        QString qsFileName = qslFileNames.at(0);
        if (!savePecFile(qsFileName))
            {
            QMessageBox::warning(this,
                tr("Aspect - Save PEC Table File"),
                tr("Failed to save PEC table to file\n\n%1")
                    .arg(qsFileName),
                QMessageBox::Ok, QMessageBox::NoButton);
            }
        }
    }


bool MainWindow::savePecFile(const QString &qsFileName)
    {
    LOG(4, "MainWindow::savePecFile");
    LOG(1, "saving PEC table \"%s\"", qsFileName.toLocal8Bit().constData());
    LOG(1, "    factor %d, aggressiveness %d",
        m_oPecTable.factor(), m_oPecTable.aggress());
    QString qsTable;
    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        {
        qsTable += QString::fromUtf8("%1").arg(m_oPecTable.valueInt(nRow));
        if (((nRow % 32) == 32 - 1) || (nRow == PecData::c_nRowNum - 1))
            {
            LOG(1, "        %s", LQS(qsTable));
            qsTable.clear();
            }
        }

    QFile oFile(qsFileName);
    // don't use flag QIODevice::Text here, because we need to write CR/LF on all platforms
    if (!oFile.open(QIODevice::WriteOnly))
        {
        LOG(1, "    ... failed: cannot open file.");
        return false;
        }
    QTextStream oStream(&oFile);
    oStream << QString::fromUtf8("Factor, Table, ReqSpeed, IsSpeed, MinSpeed, MaxSpeed\r\n");
    int nFactorAggress = m_oPecTable.fileFactorAggress();
    int nFactor = m_oPecTable.factor();
    int nReqSpeed = 0;
    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        {
        int nTable = PecSample::val2Int(m_oPecTable[nRow].m_eVal);
        if (nTable == 1)
            nReqSpeed += nFactor;
        else if (nTable == 2)
            nReqSpeed -= nFactor;
        else if (nTable == 3)
            nReqSpeed = 0;
        int nIsSpeed = nReqSpeed;
        int nMinSpeed = -47;
        int nMaxSpeed = 47;
        if (nIsSpeed < nMinSpeed)
            nIsSpeed = nMinSpeed;
        if (nIsSpeed > nMaxSpeed)
            nIsSpeed = nMaxSpeed;
        // careful:
        // write CR LF to make MCU Update Plus cope with our PEC files
        oStream << QString::fromUtf8("%1, %2, %3, %4, %5, %6\r\n")
            .arg(nFactorAggress).arg(nTable)
            .arg(nReqSpeed).arg(nIsSpeed)
            .arg(nMinSpeed).arg(nMaxSpeed);
        }
    oStream << QString::fromUtf8("\r\n");
    oFile.close();
    return true;
    }


void MainWindow::on_pecGetTable_clicked()
    {
    LOG(4, "MainWindow::on_pecGetTable_clicked");
    getPecTable();
    }


/*!
    Download PEC table from MCU via serial line

    \todo fix hang if port is not available
    \todo make port configurable
    \todo evaluate return values and do some sane error handling here ...
*/

void MainWindow::getPecTable()
    {
    LOG(4, "MainWindow::getPecTable");
    LOG(1, "downloading PEC table from MCU");

    QProgressDialog oProgress(tr("Downloading PEC table from MCU ..."),
        tr("Cancel"), 0, 1000, this);
    oProgress.setWindowTitle(tr("Aspect - Get PEC table"));
    oProgress.setMinimumDuration(0);
    oProgress.setWindowModality(Qt::WindowModal);
    QApplication::setOverrideCursor(Qt::BusyCursor);
    bool bSuccess = false;
    if (mcuCommunicationBegin())
        {
        Progress oPrg(oProgress);
        bSuccess = m_pMcuProxy->getPecTable(m_oPecTable, &oPrg);
        mcuCommunicationEnd();
        }
    QApplication::restoreOverrideCursor();
    if (bSuccess)
        {
        LOG(1, "    factor %d, aggressiveness %d",
            m_oPecTable.factor(), m_oPecTable.aggress());
        QString qsTable;
        for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
            {
            qsTable += QString::fromUtf8("%1").arg(m_oPecTable.valueInt(nRow));
            if (((nRow % 32) == 32 - 1) || (nRow == PecData::c_nRowNum - 1))
                {
                LOG(1, "        %s", LQS(qsTable));
                qsTable.clear();
                }
            }
        }
    else
        {
        LOG(1, "    failed.");
        QMessageBox::warning(this,
            tr("Aspect - Get PEC table"),
            tr("Failed to downloading PEC table from MCU.\n\n"
               "Please check that MCU is switched on and properly connected,\n"
               "and that the correct serial port is selected."),
            QMessageBox::Ok, QMessageBox::NoButton);
        }

    fillPecModel();
    recalcPeChart();
    }


void MainWindow::on_pecPutTable_clicked()
    {
    LOG(4, "MainWindow::on_pecPutTable_clicked");
    putPecTable();
    }


/*!
    Upload PEC table to MCU via serial line

    \todo show message box if opening port failed
    \todo evaluate return values and do some sane error handling here ...
    \todo query and display PEC status after uploading PEC table
    \todo verify if writing order is ok (factor, save to EEPROM, PEC values, save PEC)
*/

void MainWindow::putPecTable()
    {
    LOG(4, "MainWindow::putPecTable");
    LOG(1, "uploading PEC table to MCU");
    LOG(1, "    factor %d, aggressiveness %d",
        m_oPecTable.factor(), m_oPecTable.aggress());
    QString qsTable;
    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        {
        qsTable += QString::fromUtf8("%1").arg(m_oPecTable.valueInt(nRow));
        if (((nRow % 32) == 32 - 1) || (nRow == PecData::c_nRowNum - 1))
            {
            LOG(1, "        %s", LQS(qsTable));
            qsTable.clear();
            }
        }

    QProgressDialog oProgress(tr("Uploading PEC table to MCU ..."),
        tr("Cancel"), 0, 1000, this);
    oProgress.setWindowTitle(tr("Aspect - Put PEC table"));
    oProgress.setMinimumDuration(0);
    oProgress.setWindowModality(Qt::WindowModal);
    QApplication::setOverrideCursor(Qt::BusyCursor);
    bool bSuccess = false;
    bool bCompatible = true;
    McuDataFirmware oFirmware;
    if (mcuCommunicationBegin())
        {
        oFirmware = m_pMcuProxy->getFirmware();
        Progress oPrg(oProgress);
        // check if the PEC table is compatible with
        // the firmware found on the MCU before uploading it
        bCompatible = (
            ((oFirmware.version() < 400) && (m_oPecTable.aggress() == 0)) ||
            ((oFirmware.version() >= 400) && (m_oPecTable.aggress() > 0)));
        if (bCompatible)
            bSuccess = m_pMcuProxy->putPecTable(m_oPecTable, &oPrg);
        mcuCommunicationEnd();
        }
    QApplication::restoreOverrideCursor();
    if (bSuccess)
        {
        LOG(1, "    ok.");
        }
    else if (!bCompatible)
        {
        LOG(1, "    failed: PEC table is not compatible with firmware v%d",
            oFirmware.version() / 100);
        QMessageBox::warning(this,
            tr("Aspect - Put PEC table"),
            tr("Failed to uploading PEC table to MCU.\n\n"
               "PEC table is not compatible with firmware v%1.")
                .arg(oFirmware.version() / 100),
            QMessageBox::Ok, QMessageBox::NoButton);
        }
    else
        {
        LOG(1, "    failed.");
        QMessageBox::warning(this,
            tr("Aspect - Put PEC table"),
            tr("Failed to uploading PEC table to MCU.\n\n"
               "Please check that MCU is switched on and properly connected,\n"
               "and that the correct serial port is selected."),
            QMessageBox::Ok, QMessageBox::NoButton);
        }
    }


void MainWindow::on_pecVerifyTable_clicked()
    {
    LOG(4, "MainWindow::on_pecVerifyTable_clicked");
    verifyPecTable();
    }


/*!
    Verify PEC table against PEC table on MCU

    \todo implement
*/

void MainWindow::verifyPecTable()
    {
    LOG(4, "MainWindow::verifyPecTable");
    LOG(1, "verifying PEC table against PEC table on MCU");
    LOG(1, "    ... sorry, not yet implemented ...");
    }


void MainWindow::fillPecModel()
    {
    m_pPecModel->clear();
    m_pPecModel->setColumnCount(6);
    m_pPecModel->setRowCount(PecData::c_nRowNum);

    QStringList qslHeaders;
    qslHeaders << tr("Worm Position");
    qslHeaders << tr("PEC");
    qslHeaders << tr("acc");
    qslHeaders << tr("spd");
    qslHeaders << tr("pos");
    qslHeaders << tr("");
    m_pPecModel->setHorizontalHeaderLabels(qslHeaders);
    if (m_pUi->viewPec->header() != 0)
        m_pUi->viewPec->header()->setResizeMode(QHeaderView::ResizeToContents);

    if (m_oPecTable.aggress() == 0)
        {
        m_pUi->pecTableFirmware->setText(QString::fromUtf8("v3"));
        m_pUi->pecTableAggress->setText(QString::fromUtf8("n/a"));
        }
    else
        {
        m_pUi->pecTableFirmware->setText(QString::fromUtf8("v4"));
        m_pUi->pecTableAggress->setText(QString::fromUtf8("%1").arg(m_oPecTable.aggress()));
        }
    m_pUi->pecTableFactor->setText(QString::fromUtf8("%1").arg(m_oPecTable.factor()));
    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        {
        QStandardItem *pItem;
        pItem = new QStandardItem(QString::number(nRow));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPecModel->setItem(nRow, 0, pItem);
        pItem = new QStandardItem(QString::number(m_oPecTable.valueInt(nRow)));
        pItem->setEditable(true);
        pItem->setSelectable(true);
        m_pPecModel->setItem(nRow, 1, pItem);
        pItem = new QStandardItem(QString::number(m_oPecTable[nRow].m_nAcc));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPecModel->setItem(nRow, 2, pItem);
        pItem = new QStandardItem(QString::number(m_oPecTable[nRow].m_nSpd));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPecModel->setItem(nRow, 3, pItem);
        pItem = new QStandardItem(QString::number(m_oPecTable[nRow].m_nPos));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPecModel->setItem(nRow, 4, pItem);
        pItem = new QStandardItem(QString());
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPecModel->setItem(nRow, 5, pItem);
        }
    }


void MainWindow::changePecItem(QStandardItem *pItem)
    {
    LOG(5, "MainWindow::changePecItem");
    if ((pItem == 0) || (pItem->column() != 1))
        return;
    int nRow = pItem->row();
    if ((nRow < 0) || (nRow >= PecData::c_nRowNum))
        return;
    int nPec = pItem->text().toInt();
    if (m_oPecTable.value(nRow) == PecSample::int2Val(nPec))
        return;
    LOG(1, "manually changing PEC table item %d from %d to %d",
        nRow, m_oPecTable.valueInt(nRow), nPec);
    m_oPecTable.setValue(nRow, PecSample::int2Val(nPec));
    m_oPecTable.setFinish();
    recalcPeChart();
    }


void MainWindow::on_pecGraphOffset_valueChanged(double dfOffset)
    {
    LOG(4, "MainWindow::on_pecGraphOffset_valueChanged");
    Q_UNUSED(dfOffset)
    recalcPeChart();
    }


void MainWindow::on_pecMountQuery_clicked()
    {
    LOG(4, "MainWindow::on_pecMountQuery_clicked");
    McuDataFirmware oFirmware;
    McuDataGears oGears;
    if (mcuCommunicationBegin())
        {
        oFirmware = m_pMcuProxy->getFirmware();
        oGears = m_pMcuProxy->getGears();
        mcuCommunicationEnd();
        }
    else
        {
        LOG(0, "failed to query Firmware and Worm Wheel Teeth from MCU");
        QMessageBox::warning(this,
            tr("Aspect - MCU Communication"),
            tr("Failed to query Firmware and Worm Wheel Teeth from MCU.\n\n"
               "Please check that MCU is switched on and properly connected,\n"
               "and that the correct serial port is selected."),
            QMessageBox::Ok, QMessageBox::NoButton);
        return;
        }
    LOG(1, "querying MCU: Firmware: v%d, Worm Wheel Teeth: %d",
        oFirmware.version() / 100, oGears.raWormWheel());
    Settings::freeze();
    Settings::setMcuFirmware(oFirmware.version());
    Settings::setMcuWormWheelTeeth(oGears.raWormWheel());
    Settings::thaw();
    }


void MainWindow::on_pecGenFirmware_activated(int nFirmware)
    {
    LOG(4, "MainWindow::on_pecGenFirmware_activated");
    if (m_nFirmware / 100 == nFirmware + 3)
        return;
    LOG(1, "manually changing Firmware from v%d to v%d",
        m_nFirmware / 100, nFirmware + 3);
    Settings::setMcuFirmware(nFirmware + 3);
    }


void MainWindow::on_pecGenWormWheel_valueChanged(int nTeeth)
    {
    LOG(4, "MainWindow::on_pecWormWheel_valueChanged");
    if (m_nWormWheelTeeth == nTeeth)
        return;
    LOG(1, "manually changing Worm Wheel Teeth from %d to %d",
        m_nWormWheelTeeth, nTeeth);
    Settings::setMcuWormWheelTeeth(nTeeth);
    }


/*!
    \todo: rethink presentation and handling of parameters that affect
        more than current tab page (e.g. worm wheel teeth)
*/

void MainWindow::calcPecParams()
    {
    LOG(4, "MainWindow::calcPecParams");

    int nFirmware = Settings::getMcuFirmware();
    int nWormWheelTeeth = Settings::getMcuWormWheelTeeth();
    double dfAccel = (nFirmware < 400)? 80.: Settings::getPecGenAccel();
    double dfArcSecsPerPecPoint = 360. * 3600. / (double)nWormWheelTeeth / (double)PecData::c_nRowNum;
    double dfPecPosScale = dfArcSecsPerPecPoint / dfAccel;

    PecData::Traversal eTraversal = (Settings::getPecGenTraversal() < 0)?
        PecData::backward: PecData::forward;

    if ((m_nFirmware == nFirmware) &&
        (m_nWormWheelTeeth == nWormWheelTeeth) &&
        (m_dfPecPosScale == dfPecPosScale) &&
        (m_oPecTable.traversal() == eTraversal))
        return;

    m_nFirmware = nFirmware;
    m_nWormWheelTeeth = nWormWheelTeeth;
    m_dfPecPosScale = dfPecPosScale;
    m_oPecTable.setTraversal(eTraversal);

    LOG(3, "PEC parameters:");
    LOG(3, "    worm wheel teeth: %d", m_nWormWheelTeeth);
    LOG(3, "    PEC acceleration: 1/%0.1f (scale %f arcsec/point)", dfAccel, m_dfPecPosScale);

    m_pUi->pecGenFirmware->setCurrentIndex(nFirmware / 100 - 3);
    m_pUi->pecGenWormWheel->setValue(m_nWormWheelTeeth);
    m_pUi->mcuStatusTeeth->setText(QString::number(m_nWormWheelTeeth));
    m_pUi->mcuStatusTraversal->setText(
        QString::fromUtf8((eTraversal == PecData::forward)? "forward": "backward"));

    fillPecModel();
    recalcPeChart();
    }


void MainWindow::on_pecCalculate_clicked()
    {
    LOG(4, "MainWindow::on_pecCalculate_clicked");
    calcPecTable();
    }


/*!
    Calculate PEC table from processed PE data

    \sa PecData::calc()

    \todo make sure that processed PE data is valid
*/

void MainWindow::calcPecTable()
    {
    LOG(4, "MainWindow::calcPecTable");

    // get internal parameters for PEC table generation from settings,
    // but hardcode values for firmware older than v4.00
    double dfAccel = (m_nFirmware < 400)? 80.: Settings::getPecGenAccel();
    int nAggressMin = (m_nFirmware < 400)? 0: Settings::getPecGenAggressMin();
    int nAggressMax = (m_nFirmware < 400)? 0: Settings::getPecGenAggressMax();
    int nFactorMax = (m_nFirmware < 400)? 9: Settings::getPecGenFactorMax();

    LOG(1, "calculating PEC Table:\n"
        "    PE: file \"%s\"\n"
        "        mode %s, beg %d, end %d, %s %d\n"
        "        star %0.1f, drift %0.2f, scale %0.2f\n"
        "    PEC: fw v%d, teeth %d, traversal %s, accel 1/%0.1f (%0.3f\"/unit)\n"
        "        aggressiveness %d .. %d, factor 1 .. %d",
        LQS(m_pUi->lineEditPeFile->text()),
        (m_pUi->peProcMode->currentIndex() == 0)? "normal": "touchup",
        m_pUi->peProcBeg->value(),
        m_pUi->peProcEnd->value(),
        (m_pUi->peProcCombining->currentIndex() == 0)? "average": "median",
        m_pUi->peProcSmoothWidth->value(),
        m_pUi->peProcDecl->value(),
        m_pUi->peProcDrift->value(),
        m_pUi->peProcScale->value(),
        m_nFirmware / 100, m_nWormWheelTeeth,
        (m_oPecTable.traversal() == PecData::forward)? "forward": "backward",
        dfAccel, m_dfPecPosScale,
        nAggressMin, nAggressMax, nFactorMax);

    QProgressDialog oProgress(tr("Calculating PEC table from PE measurement ..."),
        tr("Cancel"), 0, 1000, this);
    oProgress.setWindowTitle(tr("Aspect - Calculate PEC table"));
    oProgress.setMinimumDuration(0);
    oProgress.setWindowModality(Qt::WindowModal);
    QApplication::setOverrideCursor(Qt::BusyCursor);

    // convert processed PE data into a profile for compensating the PE
    // to do so, invert the absolute PE profile:
    // - add ideal worm vs. RA ratio to measured PE data
    // - swap axis
    // - subtract ideal RA vs. worm ratio


    // center PE data around 0, add nominal gear ratio and normalize into range 0 ... PecData::c_nRowNum
    double dfMin = 1000.;
    double dfMax = -1000.;
    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        {
        if (dfMin > m_aPeProcessed.at(nRow))
            dfMin = m_aPeProcessed.at(nRow);
        if (dfMax < m_aPeProcessed.at(nRow))
            dfMax = m_aPeProcessed.at(nRow);
        }
    double dfOffset = (dfMin + dfMax) / 2.;
    double dfArcSecsPerPecPoint = 360. * 3600. / (double)m_nWormWheelTeeth / (double)PecData::c_nRowNum;
    QVector<double> aAbsProfile(PecData::c_nRowNum + 1);
    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        aAbsProfile[nRow] = (m_aPeProcessed.at(nRow) - dfOffset) / dfArcSecsPerPecPoint + (double)nRow;
    aAbsProfile[PecData::c_nRowNum] = aAbsProfile[0] + (double)PecData::c_nRowNum;

    // invert profile:
    // binary search for PEC points that are just below and above RA value,
    // then linearily interpolate between these two points
    QVector<double> aInvProfile(PecData::c_nRowNum);
    for (int nCol = 0; nCol < PecData::c_nRowNum; nCol++)
        {
        double dfColMid = (double)nCol;
        int nRowBot = 0;
        int nRowTop = PecData::c_nRowNum;
        while (nRowTop - nRowBot > 1)
            {
            int nRowMid = (nRowTop + nRowBot) / 2;
            if (dfColMid < aAbsProfile[nRowMid])
                nRowTop = nRowMid;
            else
                nRowBot = nRowMid;
            }
        double dfColBot = aAbsProfile[nRowBot];
        double dfColTop = aAbsProfile[nRowTop];
        aInvProfile[nCol] = (double)nRowBot;
        if ((dfColTop - dfColBot) > 1e-6)
            aInvProfile[nCol] += (dfColMid - dfColBot) / (dfColTop - dfColBot);
        }

    // normalize back to arcseconds and subtract nominal gear ratio
    // invert sign, as MCU firmware needs the PEC table flipped ...
    // undo centering around 0
    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        m_aPecIdeal[nRow] = (aInvProfile.at(nRow) - (double)nRow) * dfArcSecsPerPecPoint * -1. + dfOffset;

    // calculate PEC table from ideal PEC profile
    bool bSuccess = false;
    {
    Progress oPrg(oProgress);
    bSuccess = m_oPecTable.calc(m_aPecIdeal, m_dfPecPosScale,
        nAggressMin, nAggressMax, nFactorMax, &oPrg);
    }
    QApplication::restoreOverrideCursor();
    if (bSuccess)
        {
        LOG(1, "    factor %d, aggressiveness %d",
            m_oPecTable.factor(), m_oPecTable.aggress());
        QString qsTable;
        for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
            {
            qsTable += QString::fromUtf8("%1").arg(m_oPecTable.valueInt(nRow));
            if (((nRow % 32) == 32 - 1) || (nRow == PecData::c_nRowNum - 1))
                {
                LOG(1, "        %s", LQS(qsTable));
                qsTable.clear();
                }
            }
        }
    else
        {
        LOG(1, "    failed.");
        QMessageBox::warning(this,
            tr("Aspect - Generate PEC Table"),
            tr("PEC table generation failed or was canceled.\n\n"
               "If it failed, you probably need to apply more smoothing\n"
               "and/or set adequate drift compensation.\n\n"
               "Note that your PEC table was not changed"),
            QMessageBox::Ok, QMessageBox::NoButton);
        }

    fillPecModel();
    recalcPeChart();
    }




/********************************************************************/
/*
    ...
*/


/*!
    \todo rework drift compensation
        - it relies on the presence of precise timestamps
            (it timestamp lack msecs, worm pos has better resolution, but is
            proportional to time only if PE measurement was done with PEC and IntellyTrack off)
        - it probably should consider the DEC values and work with something
            more sophisticated than linear RA scaling
*/

void MainWindow::recalcPeData()
    {
    LOG(4, "MainWindow::recalcPeData");

    m_lstPePure.clear();

    double dfDrift = m_pUi->peProcDrift->value() / 60000.;
    double dfScale = m_pUi->peProcScale->value() / cos(m_pUi->peProcDecl->value() * M_PI / 180.);
    double dfOffset = m_pUi->peProcOffset->value();
    QList<PeRawSample> oData = (m_pPeRaw != 0)? m_pPeRaw->samples(): QList<PeRawSample>();
    int nBeg = m_pUi->peProcBeg->value();
    int nEnd = m_pUi->peProcEnd->value();
    if ((nEnd < 0) || (nEnd > oData.size()))
        nEnd = oData.size();
    for (int nRow = nBeg; nRow < nEnd; nRow++)
        {
        PePureSample oSample;
        oSample.m_dfWorm = oData.at(nRow).m_dfWorm;
        double dfTime = oData.at(0).m_oTime.msecsTo(oData.at(nRow).m_oTime);
        oSample.m_dfPe = (oData.at(nRow).m_dfRa - dfTime * dfDrift) * dfScale + dfOffset;
        m_lstPePure.append(oSample);
        LOG(5, "    %.03f: %.03f", oSample.m_dfWorm, oSample.m_dfPe);
        }

    // fist smooth raw sample curves for display
    bool bAverage = (m_pUi->peProcCombining->currentIndex() == 0);  // 0=Average, 1=Median
    int nSmoothWidth = m_pUi->peProcSmoothWidth->value();
    m_lstPeSmoothed.clear();
    if ((nSmoothWidth == 0) || (m_lstPePure.isEmpty()))
        {
        // if without smoothing, pass through original data
        m_lstPeSmoothed = m_lstPePure;
        }
    else
        {
        // combine samples into PEC point grid,
        // taking into account samples of nSmoothWidth PEC points

        // walk over raw data, keeping record of begin, mid and end of
        // smoothing interval:
        // - advance end of interval until hitting the end of a PEC point
        // - advance begin of interval as long as it's further from end
        //   than smoothing width
        // - calculate average/median of data samples within the interval
        //   and store at end index into list of smoothed samples
        int nRowBeg = -1;
        int nRowEnd = 0;
        int nPecMid = -1;
        int nPecBeg = -1;
        int nPecEnd = -1;
        for (int nRowMid = 0; nRowMid < m_lstPePure.size(); nRowMid++)
            {
            int nPecPrev = nPecMid;
            nPecMid = (int)floor(m_lstPePure.at(nRowMid).m_dfWorm);
            if (nPecMid == nPecPrev)
                continue;

            // advance begin of data range while data sample is outside
            // left half of smoothing interval
            // be careful to cope with both PEC table traversal directions
            // and correctly deal with PEC point wraparound and initial case
            for (;;)
                {
                int nPecDiff = abs(nPecMid - nPecBeg);
                if (nPecDiff > PecData::c_nRowNum / 2)
                    nPecDiff = abs(PecData::c_nRowNum - nPecDiff);
                if ((nPecBeg >= 0) && (nPecDiff <= nSmoothWidth / 2))
                    break;
                nRowBeg += 1;
                nPecBeg = (int)floor(m_lstPePure.at(nRowBeg).m_dfWorm);
                }
            // dito for last data sample. nRowEnd is exclusive.
            // watch of end of data, and make sure that nRowEnd is above nRowMid
            while (nRowEnd < m_lstPePure.size())
                {
                int nPecDiff = abs(nPecMid - nPecEnd);
                if (nPecDiff > PecData::c_nRowNum / 2)
                    nPecDiff = abs(PecData::c_nRowNum - nPecDiff);
                if ((nPecEnd >= 0) && (nRowEnd > nRowMid) && (nPecDiff >= (nSmoothWidth - 1) / 2))
                    break;
                nPecEnd = (int)floor(m_lstPePure.at(nRowEnd).m_dfWorm);
                nRowEnd += 1;
                }
            LOG(5, "    [%d,%d,%d[: %d,%d,%d", nRowBeg, nRowMid, nRowEnd, nPecBeg, nPecMid, nPecEnd);

            PePureSample oSample;
            oSample.m_dfWorm = (double)nPecMid;
            if (nRowEnd - nRowBeg <= 1)
                {
                // one single sample: use directly
                oSample.m_dfPe = m_lstPePure.at(nRowMid).m_dfPe;
                }
            else if (bAverage)
                {
                // Average
                double dfPe = 0.;
                for (int nRow = nRowBeg; nRow < nRowEnd; nRow++)
                    dfPe += m_lstPePure.at(nRow).m_dfPe;
                oSample.m_dfPe = dfPe / (double)(nRowEnd - nRowBeg);
                }
            else
                {
                // Median
                double dfPe = 0.;
                QList<double> oList;
                for (int nRow = nRowBeg; nRow < nRowEnd; nRow++)
                    oList.append(m_lstPePure.at(nRow).m_dfPe);
                qSort(oList);
                dfPe = oList.at((nRowEnd - nRowBeg) / 2);
                oSample.m_dfPe = dfPe;
                }
            m_lstPeSmoothed.append(oSample);
            }
        }

    // then combine all raw samples for one PEC-point and smooth again
    // don't work on smoothed data to avoid problems with gaps and such ...
    QVector<QList<double> > aPeSorted;
    aPeSorted.resize(PecData::c_nRowNum);
    for (int nRow = 0; nRow < m_lstPePure.size(); nRow++)
        {
        double dfWormPos = m_lstPePure.at(nRow).m_dfWorm;
        double dfPe = m_lstPePure.at(nRow).m_dfPe;
        int nWormPos = (int)floor(dfWormPos);
        if ((nWormPos >= 0) && (nWormPos < PecData::c_nRowNum))
            aPeSorted[nWormPos].append(dfPe);
        }
    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        {
        QList<double> lstPe;
        int nSmoothMin = - nSmoothWidth / 2;
        int nSmoothMax = (nSmoothWidth + 1) / 2;
        if (nSmoothMax < 1)
            nSmoothMax = 1;
        for (int nSmooth = nSmoothMin ; nSmooth < nSmoothMax; nSmooth++)
            lstPe += aPeSorted[(nRow + nSmooth + PecData::c_nRowNum) % PecData::c_nRowNum];
        double dfPe = 0.;
        if (lstPe.isEmpty())
            {
            // currently we rely on the user choosing adequate smoothing
            // todo: interpolate?
            }
        else if (bAverage)
            {
            // Average
            for (int nCnt = 0; nCnt < lstPe.size(); nCnt++)
                dfPe += lstPe.at(nCnt);
            dfPe /= (double)(lstPe.size());
            }
        else
            {
            // Median
            qSort(lstPe);
            dfPe = lstPe.at((lstPe.size()) / 2);
            }
        m_aPeProcessed[nRow] = dfPe;
        }

    // if in Touchup-mode, add current PEC table
    // todo: rethink this with PecIdeal
    bool bTouchup = (m_pUi->peProcMode->currentIndex() == 1);  // 0=Normal, 1=Touchup
    if (bTouchup)
        {
        for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
            m_aPeProcessed[nRow] += (double)m_oPecTable[nRow].m_nPos * m_dfPecPosScale;
        }

    // zero out PecIdeal
    // (will be recalculated when generating PEC table, as it needs worm wheel teeth
    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        m_aPecIdeal[nRow] = 0.;

    m_pPeProcessedModel->setColumnCount(2);
    m_pPeProcessedModel->setRowCount(PecData::c_nRowNum);

    QStringList qslHeaders;
    qslHeaders << tr("Worm Position");
    qslHeaders << tr("PE");
    m_pPeProcessedModel->setHorizontalHeaderLabels(qslHeaders);

    for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
        {
        QStandardItem *pItem;
        pItem = new QStandardItem(QString::number(nRow));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPeProcessedModel->setItem(nRow, 0, pItem);
        pItem = new QStandardItem(QString::number(m_aPeProcessed.at(nRow)));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPeProcessedModel->setItem(nRow, 1, pItem);
        }

    // m_lstPeriod is the list of periods.
    // each entry holds first and last sample of the period
    // (as index into the list of smoothed PE data)
    // and minimum and maximum PE value of period
    m_lstPeriod.clear();

    Period oPeriod;
    oPeriod.m_nRowBeg = -1;
    double dfWormOld = (double)PecData::c_nRowNum * 4.;
    for (int nRow = 0; nRow < m_lstPeSmoothed.size(); nRow++)
        {
        oPeriod.m_nRowEnd = nRow;
        double dfWormPos = m_lstPeSmoothed.at(nRow).m_dfWorm;
        // a leap of WormPos indicates end of current period
        if (fabs(dfWormPos - dfWormOld) > (double)PecData::c_nRowNum / 2.)
            {
            if (oPeriod.m_nRowBeg >= 0)
                m_lstPeriod.append(oPeriod);
            oPeriod.m_nRowBeg = nRow;
            oPeriod.m_dfPeMin = 1e10;
            oPeriod.m_dfPeMax = -1e10;
            }
        double dfPe = m_lstPeSmoothed.at(nRow).m_dfPe;
        if (oPeriod.m_dfPeMin > dfPe)
            oPeriod.m_dfPeMin = dfPe;
        if (oPeriod.m_dfPeMax < dfPe)
            oPeriod.m_dfPeMax = dfPe;
        dfWormOld = dfWormPos;
        }
    if (oPeriod.m_nRowBeg >= 0)
        {
        oPeriod.m_nRowEnd += 1;
        m_lstPeriod.append(oPeriod);
        }

    recalcPeChart();
    }




/********************************************************************/
/*
    PEC profile
*/


void MainWindow::on_pecProfileOffset_valueChanged(double dfOffset)
    {
    LOG(4, "MainWindow::on_pecProfileOffset_valueChanged");
    Q_UNUSED(dfOffset)
    recalcPeChart();
    }


void MainWindow::on_pecProfileDrift_valueChanged(double dfDrift)
    {
    LOG(4, "MainWindow::on_pecProfileDrift_valueChanged");
    Q_UNUSED(dfDrift)
    recalcPeChart();
    }


void MainWindow::on_pecProfileOpen_clicked()
    {
    LOG(4, "MainWindow::on_pecProfileLoad_clicked");
    QFileDialog oFileDialog(this);
    oFileDialog.setAcceptMode(QFileDialog::AcceptOpen);
    oFileDialog.setFileMode(QFileDialog::ExistingFile);
    oFileDialog.setViewMode(QFileDialog::Detail);
    oFileDialog.setWindowTitle(tr("Open PEC Profile File"));
    oFileDialog.setHistory(Settings::getPathHistory());
    oFileDialog.setDirectory(Settings::getPathDefault());
    QStringList qslFilters;
    qslFilters << tr("PEC Profile File (*.txt)");
    qslFilters << tr("All Files (*)");
    oFileDialog.setFilters(qslFilters);

    QStringList qslFileNames;
    if (oFileDialog.exec())
        {
        Settings::setPathDefault(oFileDialog.directory().absolutePath());
        qslFileNames = oFileDialog.selectedFiles();
        }
    if (qslFileNames.size() > 0)
        {
        QString qsFileName = qslFileNames.at(0);
        if (!openPecProfileFile(qsFileName))
            {
            QMessageBox::warning(this,
                tr("Aspect - Load PEC Profile"),
                tr("Failed to load PEC profile data from file\n\n%1")
                    .arg(qsFileName),
                QMessageBox::Ok, QMessageBox::NoButton);
            }
        }
    }


bool MainWindow::openPecProfileFile(const QString &qsFileName)
    {
    LOG(4, "MainWindow::openPecProfileFile");
    LOG(1, "loading PEC profile \"%s\"", qsFileName.toLocal8Bit().constData());

    QFile oFile(qsFileName);
    if (!oFile.open(QIODevice::ReadOnly | QIODevice::Text))
        {
        LOG(1, "    ... failed: cannot open file.");
        return false;
        }
    QTextStream oStream(&oFile);
    QString qsHeader = oStream.readLine();
    if (qsHeader != QString::fromUtf8("Time, Worm, Wheel, PEC"))
        {
        LOG(1, "    ... failed: wrong file format.");
        return false;
        }
    m_lstPecProfile.clear();
    while (!oStream.atEnd())
        {
        QString qsLine = oStream.readLine();
        LOG(5, "    %s", qsLine.toLocal8Bit().constData());
        QStringList qslEntryStrings = qsLine.split(",");
        PeRawSample oSample;
        if ((qslEntryStrings.size() < 4))
            continue;
        oSample.m_oTime = QTime::fromString(qslEntryStrings.at(0), QString::fromUtf8("hh:mm:ss.zzz"));
        oSample.m_dfWorm = qslEntryStrings.at(1).toDouble();
        oSample.m_dfDec = qslEntryStrings.at(2).toDouble();
        oSample.m_dfRa = qslEntryStrings.at(3).toDouble();
        m_lstPecProfile.append(oSample);
        }
    oFile.close();

    m_pPecProfileModel->clear();
    m_pPecProfileModel->setColumnCount(4);
    m_pPecProfileModel->setRowCount(m_lstPecProfile.size());

    QStringList qslHeaders;
    qslHeaders << tr("Time Stamp");
    qslHeaders << tr("Worm Pos");
    qslHeaders << tr("Wheel Pos");
    qslHeaders << tr("PEC Profile");
    m_pPecProfileModel->setHorizontalHeaderLabels(qslHeaders);

    for (int nRow = 0; nRow < m_lstPecProfile.size(); nRow++)
        {
        PeRawSample oSample = m_lstPecProfile.at(nRow);
        QStandardItem *pItem;
        pItem = new QStandardItem(oSample.m_oTime.toString(QString::fromUtf8("hh:mm:ss.zzz")));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPecProfileModel->setItem(nRow, 0, pItem);
        pItem = new QStandardItem(QString::number(oSample.m_dfWorm, 'f', 3));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPecProfileModel->setItem(nRow, 1, pItem);
        pItem = new QStandardItem(QString::number(oSample.m_dfDec, 'f', 3));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPecProfileModel->setItem(nRow, 2, pItem);
        pItem = new QStandardItem(QString::number(oSample.m_dfRa, 'f', 3));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPecProfileModel->setItem(nRow, 3, pItem);
        }

    recalcPeChart();
    return true;
    }


void MainWindow::on_pecProfileSave_clicked()
    {
    LOG(4, "MainWindow::on_pecProfileSave_clicked");
    QFileDialog oFileDialog(this);
    oFileDialog.setAcceptMode(QFileDialog::AcceptSave);
    oFileDialog.setFileMode(QFileDialog::AnyFile);
    oFileDialog.setViewMode(QFileDialog::Detail);
    oFileDialog.setWindowTitle(tr("Save PEC Profile As File"));
    oFileDialog.setHistory(Settings::getPathHistory());
    oFileDialog.setDirectory(Settings::getPathDefault());
    QStringList qslFilters;
    qslFilters << tr("PEC Profile File (*.txt)");
    qslFilters << tr("All Files (*)");
    oFileDialog.setFilters(qslFilters);
    oFileDialog.setDefaultSuffix(QString::fromUtf8("txt"));

    QStringList qslFileNames;
    if (oFileDialog.exec())
        {
        Settings::setPathDefault(oFileDialog.directory().absolutePath());
        qslFileNames = oFileDialog.selectedFiles();
        }
    if (qslFileNames.size() > 0)
        {
        QString qsFileName = qslFileNames.at(0);
        if (!savePecProfileFile(qsFileName))
            {
            QMessageBox::warning(this,
                tr("Aspect - Save PEC Profile"),
                tr("Failed to save PEC profile data to file\n\n%1")
                    .arg(qsFileName),
                QMessageBox::Ok, QMessageBox::NoButton);
            }
        }
    }


bool MainWindow::savePecProfileFile(const QString &qsFileName)
    {
    LOG(4, "MainWindow::savePecProfileFile");
    LOG(1, "saving PEC profile \"%s\"", qsFileName.toLocal8Bit().constData());

    QFile oFile(qsFileName);
    // don't use flag QIODevice::Text here, because we need to write CR/LF on all platforms
    if (!oFile.open(QIODevice::WriteOnly))
        {
        LOG(1, "    ... failed: cannot open file.");
        return false;
        }
    QTextStream oStream(&oFile);
    oStream << QString::fromUtf8("Time, Worm, Wheel, PEC\r\n");

    for (int nRow = 0; nRow < m_lstPecProfile.size(); nRow++)
        {
        PeRawSample oSample = m_lstPecProfile.at(nRow);
        oStream << QString::fromUtf8("%1, %2, %3, %4\r\n")
            .arg(oSample.m_oTime.toString(QString::fromUtf8("hh:mm:ss.zzz")))
            .arg(oSample.m_dfWorm, 0, 'f', 3)
            .arg(oSample.m_dfDec, 0, 'f', 3)
            .arg(oSample.m_dfRa, 0, 'f', 3);
        }
    oStream << QString::fromUtf8("\r\n");
    oFile.close();
    return true;
    }


void MainWindow::on_mcuGuideEast_clicked()
    {
    bool bOn = m_pUi->mcuGuideEast->isChecked();
    if (bOn && (m_eGuideDir != east))
        {
        // start moving east at guide rate
        m_eGuideDir = east;
        m_pUi->mcuGuideWest->setChecked(false);
        if (mcuCommunicationBegin())
            {
            m_pMcuProxy->guideOff();
            m_pMcuProxy->guideEast();
            mcuCommunicationEnd();
            }
        }
    else if ((!bOn) && (m_eGuideDir == east))
        {
        // stop moving
        m_eGuideDir = stop;
        if (mcuCommunicationBegin())
            {
            m_pMcuProxy->guideOff();
            mcuCommunicationEnd();
            }
        }
    }


void MainWindow::on_mcuGuideWest_clicked()
    {
    bool bOn = m_pUi->mcuGuideWest->isChecked();
    if (bOn && (m_eGuideDir != west))
        {
        // start moving west at guide rate
        m_eGuideDir = west;
        m_pUi->mcuGuideEast->setChecked(false);
        if (mcuCommunicationBegin())
            {
            m_pMcuProxy->guideOff();
            m_pMcuProxy->guideWest();
            mcuCommunicationEnd();
            }
        }
    else if ((!bOn) && (m_eGuideDir == west))
        {
        // stop moving
        m_eGuideDir = stop;
        if (mcuCommunicationBegin())
            {
            m_pMcuProxy->guideOff();
            mcuCommunicationEnd();
            }
        }
    }


void MainWindow::on_mcuStart_clicked()
    {
    LOG(4, "MainWindow::on_mcuStart_clicked");

    m_pUi->mcuStart->setEnabled(false);
    m_pUi->mcuPause->setText(QString::fromUtf8("Pause"));
    m_pUi->mcuPause->setEnabled(false);
    m_pUi->mcuPause->setChecked(false);

    if (m_pUi->mcuStart->isChecked())
        {
        clearPecProfile();
        if (startPecProfile(true))
            {
            m_pUi->mcuStart->setText(QString::fromUtf8("Stop"));
            m_pUi->mcuPause->setEnabled(true);
            }
        else
            {
            m_pUi->mcuStart->setText(QString::fromUtf8("Start"));
            m_pUi->mcuStart->setChecked(false);
            }
        }
    else
        {
        stopPecProfile();
        m_pUi->mcuStart->setText(QString::fromUtf8("Start"));
        }
    m_pUi->mcuStart->setEnabled(true);
    }


void MainWindow::on_mcuPause_clicked()
    {
    LOG(4, "MainWindow::on_mcuPause_clicked");

    if (!m_pUi->mcuStart->isChecked())
        {
        m_pUi->mcuPause->setText(QString::fromUtf8("Pause"));
        m_pUi->mcuPause->setChecked(false);
        m_pUi->mcuPause->setEnabled(false);
        return;
        }

    m_pUi->mcuPause->setEnabled(false);
    if (m_pUi->mcuPause->isChecked())
        {
        stopPecProfile();
        m_pUi->mcuPause->setText(QString::fromUtf8("Continue"));
        m_pUi->mcuPause->setEnabled(true);
        }
    else
        {
        m_pUi->mcuPause->setChecked(false);
        if (startPecProfile(false))
            {
            m_pUi->mcuPause->setText(QString::fromUtf8("Pause"));
            m_pUi->mcuPause->setEnabled(true);
            }
        else
            {
            // stop PEC profiling completely, as resuming after
            // e.g. switching the MCU off and on doesn't make sense
            // (time and worm count ran out of sync)
            m_pUi->mcuStart->setText(QString::fromUtf8("Start"));
            m_pUi->mcuStart->setChecked(false);
            m_pUi->mcuStart->setEnabled(true);
            m_pUi->mcuPause->setText(QString::fromUtf8("Pause"));
            m_pUi->mcuPause->setChecked(false);
            m_pUi->mcuPause->setEnabled(false);
            }
        }
    }


void MainWindow::clearPecProfile()
    {
    // clear PEC profile listview and graph
    m_lstPecProfile.clear();

    m_pPecProfileModel->clear();
    m_pPecProfileModel->setColumnCount(4);

    QStringList qslHeaders;
    qslHeaders << tr("Time Stamp");
    qslHeaders << tr("Worm Pos");
    qslHeaders << tr("Wheel Pos");
    qslHeaders << tr("PEC Profile");
    m_pPecProfileModel->setHorizontalHeaderLabels(qslHeaders);

    recalcPeChart();
    }


bool MainWindow::startPecProfile(bool bSetSpeed)
    {
    bool bSuccess = false;
    McuDataFirmware oFirmware;
    McuDataGears oGears;
    if (mcuCommunicationBegin())
        {
        oFirmware = m_pMcuProxy->getFirmware();
        oGears = m_pMcuProxy->getGears();
        LOG(1, "querying MCU: Firmware: v%d, Worm Wheel Teeth: %d",
            oFirmware.version() / 100, oGears.raWormWheel());
        if (bSetSpeed)
            {
            // switch to sidereal tracking rate, and switch intellytrack off
            bSuccess =
                m_pMcuProxy->setTrackingRate(McuProxy::sidereal) &&
                m_pMcuProxy->setIntellyTrack(false);
            }
        else
            bSuccess = true;
        }
    if (bSuccess)
        {
        Settings::freeze();
        Settings::setMcuFirmware(oFirmware.version());
        Settings::setMcuWormWheelTeeth(oGears.raWormWheel());
        Settings::thaw();
        LOG(1, "starting PEC profiling");
        m_pMcuTimer->start((int)(m_pUi->mcuPoll->value() * 1000.));
        }
    else
        {
        LOG(0, "failed to start PEC profiling");
        QMessageBox::warning(this,
            tr("Aspect - PEC Profiling"),
            tr("Failed to connect to MCU.\n\n"
            "Please check that MCU is switched on and properly connected,\n"
            "and that the correct serial port is selected."),
            QMessageBox::Ok, QMessageBox::NoButton);
        }
    return bSuccess;
    }


void MainWindow::stopPecProfile()
    {
    m_pMcuTimer->stop();
    mcuCommunicationEnd();
    LOG(1, "stopping PEC profiling");
    }


void MainWindow::mcuTimer()
    {
    LOG(4, "MainWindow::mcuTimer");

    McuDataPecState oPecState = m_pMcuProxy->getPecState();
    m_pUi->mcuStatusPec->setText(oPecState.state());
    if (oPecState.pec() == 5)
        {
        LOG(0, "failed to query PEC state while PEC profiling");

        m_pUi->mcuStart->setEnabled(false);
        m_pUi->mcuPause->setText(QString::fromUtf8("Pause"));
        m_pUi->mcuPause->setEnabled(false);
        m_pUi->mcuPause->setChecked(false);
        stopPecProfile();
        QMessageBox::warning(this,
            tr("Aspect - PEC Profiling"),
            tr("Failed to query PEC state from MCU.\n\n"
            "Please check that MCU is switched on and properly connected,\n"
            "and that the correct serial port is selected."),
            QMessageBox::Ok, QMessageBox::NoButton);

        m_pUi->mcuStart->setChecked(false);
        m_pUi->mcuStart->setText(QString::fromUtf8("Start"));
        m_pUi->mcuStart->setEnabled(true);
        return;
        }

    PeRawSample oSample;
    oSample.m_oTime = oPecState.time();
    oSample.m_dfWorm = oPecState.wormPos();
    if (m_lstPecProfile.size() == 0)
        {
        oSample.m_dfRa = 0.;
        oSample.m_dfDec = 0.;
        }
    else
        {
        double dfWormWheelTeeth = (double)m_nWormWheelTeeth;
        double dfTime = (double)(m_lstPecProfile.at(0).m_oTime.msecsTo(oSample.m_oTime)) / 1000.;
        double dfWorm = (oSample.m_dfWorm - m_lstPecProfile.at(0).m_dfWorm) * (-1.);
        dfWorm -= floor(dfWorm / (double)PecData::c_nRowNum) * (double)PecData::c_nRowNum;
        double dfWheel = dfTime / (m_dfSiderealDay / dfWormWheelTeeth / (double)PecData::c_nRowNum);
        dfWheel -= floor(dfWheel / (double)PecData::c_nRowNum) * (double)PecData::c_nRowNum;
        double dfWormError = dfWorm - dfWheel;
        dfWormError -= floor(dfWormError / (double)PecData::c_nRowNum + .5) * (double)PecData::c_nRowNum;
        oSample.m_dfRa = (dfWormError) * (360. * 3600. / dfWormWheelTeeth / (double)PecData::c_nRowNum);
        oSample.m_dfDec = dfWheel; // abuse Dec as storage for wheel position
        }
    m_lstPecProfile.append(oSample);

    QList<QStandardItem *> oItemList;
    oItemList.append(new QStandardItem(oSample.m_oTime.toString(QString::fromUtf8("hh:mm:ss.zzz"))));
    oItemList.append(new QStandardItem(QString::number(oSample.m_dfWorm, 'f', 3)));
    oItemList.append(new QStandardItem(QString::number(oSample.m_dfDec, 'f', 3)));
    oItemList.append(new QStandardItem(QString::number(oSample.m_dfRa, 'f', 3)));
    m_pPecProfileModel->appendRow(oItemList);
    m_pUi->viewPecProfile->scrollTo(oItemList.at(0)->index());
    LOG(3, "%s %.3f %.3f %.3f",
        LQS(oSample.m_oTime.toString(QString::fromUtf8("hh:mm:ss.zzz"))),
        oSample.m_dfWorm, oSample.m_dfDec, oSample.m_dfRa);

    recalcPeChart();
    }




/********************************************************************/
/*
    Charts
*/


/*!
    Recalculate and redraw all kind of charts

    \todo move this out of MainWindow
    \todo implement PEC profile graph properly
*/

void MainWindow::recalcPeChart()
    {
    LOG(4, "MainWindow::recalcPeChart");

    m_pUi->chartPe->clear();

    bool bChartPeRaw = m_pUi->toggleChartPeRaw->isChecked();
    bool bChartPeProcessed = m_pUi->toggleChartPeProcessed->isChecked();
    bool bChartPecIdeal = m_pUi->toggleChartPecIdeal->isChecked();
    bool bChartPecTable = m_pUi->toggleChartPecTable->isChecked();
    bool bChartPecProfile = m_pUi->toggleChartPecProfile->isChecked();

    double dfPecOffset = -m_pUi->pecGraphOffset->value() / m_dfPecPosScale;
    Graph oGraphPeRaw(Graph::line);
    Graph oGraphPeProcessed(Graph::line);
    Graph oGraphPecIdeal(Graph::line);
    Graph oGraphPos(Graph::line);
    Graph oGraphSpd((m_oPecTable.traversal() == PecData::forward)?
        Graph::stairLeft: Graph::stairRight);
    Graph oGraphAcc(Graph::tick);
    Graph oGraphPecProfile(Graph::line);

    Grid oGridPe;
    Grid oGridPecPos;
    Grid oGridPecSpd;
    Grid oGridPecAcc;

    oGridPe.addData(0., 0.);
    oGridPe.addData((double)PecData::c_nRowNum, 0.);
    oGridPecSpd.addData(0., -(double)PecData::c_nFactorNum);
    oGridPecSpd.addData((double)PecData::c_nRowNum, (double)PecData::c_nFactorNum);
    oGridPecSpd.addGraph(oGraphSpd);
    oGridPecAcc.addData(0., -1.);
    oGridPecAcc.addData((double)PecData::c_nRowNum, 1.);
    oGridPecAcc.addGraph(oGraphAcc);

    if (bChartPeRaw)
        {
        for (int nPeriod = 0; nPeriod < m_lstPeriod.size(); nPeriod++)
            {
            int nRowBeg = m_lstPeriod.at(nPeriod).m_nRowBeg;
            int nRowEnd = m_lstPeriod.at(nPeriod).m_nRowEnd;
            LOG(4, "MainWindow::recalcPeChart: period %d: [%d..%d]", nPeriod, nRowBeg, nRowEnd);
            for (int nRow = nRowBeg; nRow < nRowEnd; nRow++)
                oGraphPeRaw.addData(
                    m_lstPeSmoothed.at(nRow).m_dfWorm,
                    m_lstPeSmoothed.at(nRow).m_dfPe,
                    (nRow == nRowBeg));
            }
        oGridPe.addGraph(oGraphPeRaw);
        }

    if (bChartPeProcessed)
        {
        for (int nRow = 0; nRow <= PecData::c_nRowNum; nRow++)
            oGraphPeProcessed.addData((double)nRow, m_aPeProcessed.at(nRow % PecData::c_nRowNum));
        oGridPe.addGraph(oGraphPeProcessed);
        }

    if (bChartPecIdeal)
        {
        for (int nRow = 0; nRow <= PecData::c_nRowNum; nRow++)
            oGraphPecIdeal.addData((double)nRow, m_aPecIdeal.at(nRow % PecData::c_nRowNum));
        oGridPe.addGraph(oGraphPecIdeal);
        }

    if (bChartPecTable)
        {
        for (int nRow = 0; nRow < PecData::c_nRowNum; nRow++)
            {
            oGraphPos.addData((double)nRow,
                (double)m_oPecTable[nRow].m_nPos * m_dfPecPosScale - dfPecOffset);
            oGraphSpd.addData((double)nRow, (double)m_oPecTable[nRow].m_nSpd);
            switch (m_oPecTable[nRow].m_eVal)
                {
                case PecSample::keep:
                    break;
                case PecSample::faster:
                    oGraphAcc.addData((double)nRow + .5, .75);
                    break;
                case PecSample::slower:
                    oGraphAcc.addData((double)nRow + .5, -.75);
                    break;
                case PecSample::reset:
                    oGraphAcc.addData((double)nRow + .5, 0.);
                    break;
                }
            }
        oGraphPos.addData((double)PecData::c_nRowNum,
            (double)m_oPecTable[0].m_nPos * m_dfPecPosScale - dfPecOffset);
        oGraphSpd.addData((double)PecData::c_nRowNum, (double)m_oPecTable[0].m_nSpd);
        oGridPe.addGraph(oGraphPos);
        oGridPecPos.addData((double)PecData::c_nRowNum, 0.);
        oGridPecPos.addGraph(oGraphPos);
        }

    if ((bChartPecProfile) && (m_lstPecProfile.size() > 0))
        {
        int nRow = (int)floor(m_lstPecProfile.at(0).m_dfWorm);
        double dfPecProfileOffset = ((double)m_oPecTable[nRow].m_nPos + dfPecOffset) * m_dfPecPosScale;
        double dfXStart = (m_lstPecProfile.size() > 0)? m_lstPecProfile.at(0).m_dfWorm: 0.;
        double dfXPrev = dfXStart;
        double dfXPeriod = 0.;
        for (int nRow = 0; nRow < m_lstPecProfile.size(); nRow++)
            {
            double dfX = m_lstPecProfile.at(nRow).m_dfWorm;
            bool bWrap = (fabs(dfX - dfXPrev) > (double)PecData::c_nRowNum / 2.);
            if (bWrap)
                dfXPeriod += (double)PecData::c_nRowNum * ((dfX > dfXPrev)? -1.: 1.);
            double dfY = dfPecProfileOffset + m_pUi->pecProfileOffset->value() -
                m_lstPecProfile.at(nRow).m_dfRa -
                (dfX + dfXPeriod - dfXStart) / (double)PecData::c_nRowNum *
                    m_pUi->pecProfileDrift->value() / 60. * m_dfSiderealDay / m_nWormWheelTeeth ;
            oGraphPecProfile.addData(dfX, dfY, bWrap);
            dfXPrev = dfX;
            }
        oGridPe.addData((double)PecData::c_nRowNum, 0.);
        oGridPe.addGraph(oGraphPecProfile);
        }

    // limit y range of main chart to give good appearance and
    // avoid insane y values hanging the QGraphicsView
    double dfRangeX = oGridPe.maxX() - oGridPe.minX();
    double dfRangeY = oGridPe.maxYSanitized() - oGridPe.minYSanitized();
    // scale main chart to same x/y resolution, so that numbers look good
    // move main chart down such that top edge is at scene coordinate 0
    static const double c_dfScale = 450.;
    static const double c_dfBorder = 25.;
    double dfXScale = c_dfScale / dfRangeX;
    double dfYScalePos = c_dfScale / dfRangeY;
    double dfYOffsPos = oGridPe.maxYSanitized();
    // scale speed chart to same x/y resolution
    // move speed chart below main chart, considering that maximum value is PecData::c_nFactorNum
    double dfYScaleSpd = c_dfScale / dfRangeX;
    double dfYOffsSpd = (c_dfScale + c_dfBorder) / dfYScaleSpd + (double)PecData::c_nFactorNum;
    // scale acceleration chart to four times x/y resolution (for readablility)
    // move acceleration chart below speed chart, considering that maximum value is 1
    double dfYScaleAcc = c_dfScale / dfRangeX * 4.;
    double dfYOffsAcc = (c_dfScale + c_dfBorder + dfYScaleSpd * 2. * (double)PecData::c_nFactorNum +
        c_dfBorder) / dfYScaleAcc + 1.;

    oGraphPeRaw.render(m_pPeChart,
        dfXScale, dfYOffsPos, dfYScalePos,
        2.);
    oGraphPeProcessed.render(m_pPeChart,
        dfXScale, dfYOffsPos, dfYScalePos,
        2.5, QPen(Qt::red));
    oGraphPecIdeal.render(m_pPeChart,
        dfXScale, dfYOffsPos, dfYScalePos,
        3., QPen(QColor(255, 170, 0)));
    oGraphPos.render(m_pPeChart,
        dfXScale, dfYOffsPos, dfYScalePos,
        4., QPen(Qt::darkGreen));
    oGraphSpd.render(m_pPeChart,
        dfXScale, dfYOffsSpd, dfYScaleSpd,
        4., QPen(Qt::darkGreen));
    oGraphAcc.render(m_pPeChart,
        dfXScale, dfYOffsAcc, dfYScaleAcc,
        4., QPen(Qt::darkGreen));
    oGraphPecProfile.render(m_pPeChart,
        dfXScale, dfYOffsPos, dfYScalePos,
        3.5, QPen(QColor(170, 0, 255)));

    oGridPe.render(m_pPeChart,
        dfXScale, dfYOffsPos, dfYScalePos,
        Grid::both, Grid::both, Grid::min, Grid::min,
        32., 8., 5., 1.);
    oGridPecPos.render(m_pPeChart,
        dfXScale, dfYOffsPos, dfYScalePos,
        Grid::none, Grid::none, Grid::undec, Grid::max,
        32., 8., 5., 1., -1., -2., Qt::darkGreen);
    oGridPecSpd.render(m_pPeChart,
        dfXScale, dfYOffsSpd, dfYScaleSpd,
        Grid::both, Grid::minor, Grid::min, Grid::max,
        32., 8., (double)PecData::c_nFactorNum, (double)PecData::c_nFactorNum,
        -1., -2., Qt::darkGreen);
    oGridPecAcc.render(m_pPeChart,
        dfXScale, dfYOffsAcc, dfYScaleAcc,
        Grid::both, Grid::minor, Grid::undec, Grid::max,
        32., 8., 5., 1., -1., -2., Qt::darkGreen);

    m_pUi->chartPe->requestSceneRender();
    }

