
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2007 by Wolfgang Hoffmann <woho@woho.de>                *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/


#include "logwindow.h"
#include "ui_logwindow.h"
#include "log.h"

#include <QtGui>




/********************************************************************/
/*!
    \class LogDialog
    \brief Dialog for viewing log of communication with MCU

    \todo Fix autoscrolling if user klicked in or moved with
        the cursor key, and then gets back to the last line

    \todo As QTextEdit seems to load CPU quite heavily with
        high maximumBlockCount, implement a fast ringbuffer here
        that is copied into QTextEdit with load balancing, and
        only if the window is visible. Is that worth the effort?
*/

LogDialog::LogDialog(QWidget *pParent)
:   QDialog(pParent)
    {
    setAttribute(Qt::WA_DeleteOnClose, false);
    setAttribute(Qt::WA_QuitOnClose, false);

    m_bAtEnd = true;

    m_pUi = new Ui::LogDialog;
    m_pUi->setupUi(this);

    QTextDocument *pDoc = m_pUi->logView->document();
    // keep small for now to avoid performance hit
    pDoc->setMaximumBlockCount(40);

    /*
    // somehow this won't work ...
    QFont oFixedFont = pDoc->defaultFont();
    oFixedFont.setFixedPitch(QFont::TypeWriter);
    pDoc->setDefaultFont(oFixedFont);
    */

    // append a line with single spage to let user go below last log line for autoscroll
    // (for some strange reason, the space is needed ...)
    m_pUi->logView->insertPlainText(QString::fromUtf8("\n "));
    // copy text cursor and set to start position for appending log lines
    m_oAppendCursor = m_pUi->logView->textCursor();
    m_oAppendCursor.movePosition(QTextCursor::Start);

    resize(800, 600);
    }


LogDialog::~LogDialog()
    {
    delete m_pUi;
    }


void LogDialog::on_logView_cursorPositionChanged()
    {
    m_bAtEnd = m_pUi->logView->textCursor().atEnd();
    }


void LogDialog::append(const QString &qsText)
    {
    QString qsTime = QTime::currentTime().toString(QString::fromUtf8("hh:mm:ss:zzz "));
    QStringList qslLines = qsText.split(QString::fromUtf8("\n"));
    for (int nLine = 0; nLine < qslLines.size(); nLine++)
        {
        QString qsLine = qslLines.at(nLine);
        // insert timestamp if at begin of line and not at empty last line
        if ((m_oAppendCursor.columnNumber() == 0) &&
            ((nLine < qslLines.size() - 1) || (!qsLine.isEmpty())))
            m_oAppendCursor.insertText(qsTime);
        m_oAppendCursor.insertText(qsLine);
        // insert line break if not at last line
        // QString::split() by default keeps empty entries, i.e. if
        // original text ended with line break, we get an additional
        // empty string as last entry of qslLines
        if (nLine < qslLines.size() - 1)
            m_oAppendCursor.insertText(QString::fromUtf8("\n"));
        }

    // eventually scroll such that user-cursor is visible
    if (m_bAtEnd)
        {
        m_pUi->logView->moveCursor(QTextCursor::End);
        m_pUi->logView->ensureCursorVisible();
        }
    }

