
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2007 by Wolfgang Hoffmann <woho@woho.de>                *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/

#include "log.h"
#include "settings.h"

#include <QtCore>




/********************************************************************/
/*!
    \class LogImpl
    \brief Internal logging implementation
*/


LogImpl LogImpl::g_oLog(QString::fromUtf8("log"));
LogImpl LogImpl::g_oMcu(QString::fromUtf8("mcu"));


LogImpl::LogImpl(const QString &qsType)
:   m_qsType(qsType),
    m_pFile(0),
    m_bFirstStartup(true)
    {
    m_oTimerFlush.setSingleShot(true);
    m_oTimerClose.setSingleShot(true);
    connect(&m_oTimerFlush, SIGNAL(timeout()), this, SLOT(timeoutFlush()));
    connect(&m_oTimerClose, SIGNAL(timeout()), this, SLOT(timeoutClose()));
    }


void LogImpl::startup()
    {
    if (m_pFile != 0)
        return;

    // defer querying and connecting Settings until static constructors are done.
    m_oPath = Settings::getPathLog();
    m_bDoLog = (m_qsType == QString::fromUtf8("log"))?
        Settings::getLoggingGeneral(): Settings::getLoggingMcu();
    if (m_bFirstStartup)
        Settings::connect(this, SLOT(settingsChanged()));
    m_bFirstStartup = false;

    QDateTime oDate = QDateTime::currentDateTime();
    QString qsDate = oDate.toString(QString::fromUtf8("yyyyMMdd"));
    m_pFile = new QFile(m_oPath.filePath(
        QString::fromUtf8("aspect_%1_%2.txt").arg(m_qsType).arg(qsDate)));
    m_bLineComplete = true;

    QString qsTime = oDate.toString(QString::fromUtf8("dd.MM.yyyy hh:mm:ss"));
    append(QString::fromUtf8("\n############################################################\n"));
    append(QString::fromUtf8("####  started %1\n").arg(qsTime));
    append(QString::fromUtf8("############################################################\n\n"));
    }


void LogImpl::shutdown()
    {
    if (m_pFile == 0)
        return;

    QDateTime oDate = QDateTime::currentDateTime();
    QString qsTime = oDate.toString(QString::fromUtf8("dd.MM.yyyy hh:mm:ss"));
    append(QString::fromUtf8("\n############################################################\n"));
    append(QString::fromUtf8("####  finished %1\n").arg(qsTime));
    append(QString::fromUtf8("############################################################\n\n"));
    m_oTimerFlush.stop();
    m_oTimerClose.stop();
    if (m_pFile->isOpen())
        m_pFile->close();
    delete m_pFile;
    m_pFile = 0;
    }


void LogImpl::settingsChanged()
    {
    if (m_oPath != Settings::getPathLog())
        {
        shutdown();
        startup();
        }
    bool bDoLog = (m_qsType == QString::fromUtf8("log"))?
        Settings::getLoggingGeneral(): Settings::getLoggingMcu();
    if (m_bDoLog != bDoLog)
        {
        if (!bDoLog)
            {
            QDateTime oDate = QDateTime::currentDateTime();
            QString qsTime = oDate.toString(QString::fromUtf8("dd.MM.yyyy hh:mm:ss"));
            append(QString::fromUtf8("\n############################################################\n"));
            append(QString::fromUtf8("####  logging stopped %1\n").arg(qsTime));
            append(QString::fromUtf8("############################################################\n\n"));
            }
        m_bDoLog = bDoLog;
        if (m_bDoLog)
            {
            QDateTime oDate = QDateTime::currentDateTime();
            QString qsTime = oDate.toString(QString::fromUtf8("dd.MM.yyyy hh:mm:ss"));
            append(QString::fromUtf8("\n############################################################\n"));
            append(QString::fromUtf8("####  logging started %1\n").arg(qsTime));
            append(QString::fromUtf8("############################################################\n\n"));
            }
        }
    }


void LogImpl::append(const QString &qsText, const QTime &oTime)
    {
    if (m_pFile == 0)
        return;
    if (!m_bDoLog)
        return;
    if ((!m_pFile->isOpen()) &&
        (!m_pFile->open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Append)))
        return;
    {
    QTextStream oStream(m_pFile);
    QString qsTime = oTime.toString(QString::fromUtf8("hh:mm:ss:zzz "));

    // QString::split() by default keeps empty entries, i.e. if
    // original text ended with line break, we get an additional
    // empty string as last entry of qslLines
    QStringList qslLines = qsText.split(QString::fromUtf8("\n"));
    for (int nLine = 0; nLine < qslLines.size() - 1; nLine++)
        {
        // insert timestamp if at begin of line
        if (m_bLineComplete)
            oStream << qsTime;
        oStream << qslLines.at(nLine) << QString::fromUtf8("\n");
        m_bLineComplete = true;
        }
    if (!qslLines.at(qslLines.size() - 1).isEmpty())
        {
        // insert timestamp if at begin of line
        if (m_bLineComplete)
            oStream << qsTime;
        oStream << qslLines.at(qslLines.size() - 1);
        m_bLineComplete = false;
        }
    }
    if (!m_oTimerFlush.isActive())
        m_oTimerFlush.start(1000);
    m_oTimerClose.start(3000);
    }


void LogImpl::timeoutFlush()
    {
    if (m_pFile == 0)
        return;
    if (m_pFile->isOpen())
        m_pFile->flush();
    }


void LogImpl::timeoutClose()
    {
    if (m_pFile == 0)
        return;
    if (m_pFile->isOpen())
        m_pFile->close();
    }




/********************************************************************/
/*!
    \class Log
    \brief Logging to console and temporary log file
*/

void Log::mcu(const QString &qsText, const QTime &oTime)
    {
    LogImpl::g_oMcu.append(qsText, oTime);
    }


void Log::log(const char *szFmt, ...)
    {
    // clipboarded from qDebug()
    static const size_t c_nBufSize = 8193;
    char szBuf[c_nBufSize];
    szBuf[c_nBufSize - 1] = '\0';
    va_list oArgs;
    va_start(oArgs, szFmt);
    qvsnprintf(szBuf, c_nBufSize - 1, szFmt, oArgs);
    va_end(oArgs);
    //LogImpl::g_oWieSchoen.appendLog(QString::fromLocal8Bit(szBuf));
    LogImpl::g_oLog.append(QString::fromUtf8(szBuf) + QString::fromUtf8("\n"));
    }


void Log::startup()
    {
    LogImpl::g_oLog.startup();
    LogImpl::g_oMcu.startup();
    }


void Log::shutdown()
    {
    LogImpl::g_oLog.shutdown();
    LogImpl::g_oMcu.shutdown();
    }

