
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2006-2007 by Wolfgang Hoffmann <woho@woho.de>           *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/


#include "datapemcu.h"
#include "ui_datapemcu.h"
#include "log.h"

#include <QtGui>
#include <math.h>


DataPeMcu::DataPeMcu(QWidget *pParent)
:   QWidget(pParent),
    m_bInModelRefill(false)
    {
    m_pUi = new Ui::DataPeMcu;
    m_pUi->setupUi(this);

    m_pPeModel = new QStandardItemModel(this);
    m_pUi->viewPe->setModel(m_pPeModel);

    connect(m_pPeModel, SIGNAL(itemChanged(QStandardItem *)), this, SLOT(changePeItem(QStandardItem *)));
    connect(&m_oPoller, SIGNAL(timeout()), this, SLOT(readPeFile()));
    }


DataPeMcu::~DataPeMcu()
    {
    if (m_oPeFile.isOpen())
        m_oPeFile.close();
    delete m_pUi;
    }


void DataPeMcu::clearData()
    {
    pollPeFile(false);
    clearPeData();
    }


bool DataPeMcu::probePeFile(const QString &qsFileName) const
    {
    LOG(4, "DataPeMcu::probePeFile(%s)", LQS(qsFileName));

    bool bIsMcuFormat = false;
    QFile oPeFile(qsFileName);
    if (oPeFile.open(QIODevice::ReadOnly | QIODevice::Text))
        {
        QString qsHeader = QString::fromLocal8Bit(oPeFile.readLine()).trimmed();
        LOG(5, "    %s", LQS(qsHeader));
        if (qsHeader == QString::fromUtf8("Time, PEC Table, RA Error, DEC Error"))
            bIsMcuFormat = true;
        oPeFile.close();
        }

    return bIsMcuFormat;
    }


void DataPeMcu::setPeFileName(const QString &qsFileName)
    {
    if (m_oPeFile.isOpen())
        m_oPeFile.close();
    m_oPeFile.setFileName(qsFileName);
    }


bool DataPeMcu::loadPeFile()
    {
    LOG(4, "DataPeMcu::loadPeFile(%s)", LQS(m_oPeFile.fileName()));

    if (m_oPeFile.isOpen())
        m_oPeFile.close();

    clearData();
    if (!openPeFile())
        return false;
    readPeFile();
    closePeFile();

    return true;
    }


bool DataPeMcu::pollPeFile(bool bOn)
    {
    if (bOn)
        {
        // verify that polling was off
        if (m_oPoller.isActive())
            return false;
        clearData();
        if (!openPeFile())
            return false;
        m_oPoller.start(1000);
        m_qsLineFragment = QString();
        readPeFile();
        }
    else
        {
        // verify that polling was on
        if (!m_oPoller.isActive())
            return false;
        m_oPoller.stop();
        closePeFile();
        }
    return true;
    }


/*!
    Clear data list m_lstPeRaw and model m_pPeModel.

    Open PE data file, asuming that filename on m_oPeFile is already
    set. Set device to text stream and read and check file header.
*/

bool DataPeMcu::openPeFile()
    {
    if (!m_oPeFile.open(QIODevice::ReadOnly | QIODevice::Text))
        {
        LOG(0, "DataPeMcu::loadPeFile(%s): failed to open file", LQS(m_oPeFile.fileName()));
        return false;
        }
    QString qsHeader = QString::fromLocal8Bit(m_oPeFile.readLine()).trimmed();
    if (qsHeader != QString::fromUtf8("Time, PEC Table, RA Error, DEC Error"))
        {
        LOG(0, "DataPeMcu::loadPeFile(%s): wrong file format:", LQS(m_oPeFile.fileName()));
        LOG(0, "    \"%s\"", LQS(qsHeader));
        return false;
        }

    return true;
    }


void DataPeMcu::closePeFile()
    {
    if (!m_oPeFile.isOpen())
        return;
    m_oPeFile.close();
    }


/*!
    Clear m_lstPeRaw and clear and reset model m_pPeModel
*/

void DataPeMcu::clearPeData()
    {
    m_pPeModel->clear();

    m_lstPeRaw.clear();
    m_pPeModel->setColumnCount(4);
    QStringList qslHeaders;
    qslHeaders << tr("Time");
    qslHeaders << tr("Worm Position");
    qslHeaders << tr("RA Error");
    qslHeaders << tr("DEC Error");
    m_pPeModel->setHorizontalHeaderLabels(qslHeaders);
    }


/*!
    Read lines from file, as long as complete lines of data
    are available.

    Enter data into m_lstPeRaw while reading, and update
    model m_pPeModel after reading is done. Finally emit
    dataPeChanged().

    Don't use QTextStream, as it obviously doesn't update
    it's internal buffer when data is appended to a file.
*/

void DataPeMcu::readPeFile()
    {
    LOG(5, "readPeFile()");
    while (m_oPeFile.bytesAvailable() > 0)
        {
        QByteArray baLine = m_oPeFile.readLine();
        // careful:
        // if QByteArray is empty, no more complete lines available.
        // otherwise if QString is empty, skip empty line
        if (baLine.isEmpty())
            break;
        // need to keep a simple buffer for keeping incomplete lines accross readPeFile() calls
        m_qsLineFragment += QString::fromLocal8Bit(baLine).trimmed();
        if (!baLine.endsWith('\n'))
            break;
        QString qsLine = m_qsLineFragment;
        m_qsLineFragment = QString();
        LOG(5, "    \"%s\"", LQS(qsLine));
        if (qsLine.isEmpty())
            continue;
        QStringList qslEntryStrings = qsLine.split(",");
        PeRawSample oSample;
        oSample.m_oTime = QTime();
        if (qslEntryStrings.size() > 4)
            {
            // cope with files written under German language settings,
            // where decimal separators '.' becomes ','
            // heuristically convert every second ',' back to '.',
            // starting from backwards, as timestamp might have no decimals
            int nNum = 0;
            for (int nPos = qsLine.lastIndexOf(','); nPos > 0; nPos = qsLine.lastIndexOf(',', nPos-1))
                {
                if ((nNum % 2) == 0)
                    qsLine.replace(nPos, 1, '.');
                nNum++;
                }
            qslEntryStrings = qsLine.split(",");
            }
        if (qslEntryStrings.size() > 0)
            {
            oSample.m_oTime = QTime::fromString(qslEntryStrings.at(0), QString::fromUtf8("hh:mm:ss.zzz"));
            if (!oSample.m_oTime.isValid())
                oSample.m_oTime = QTime::fromString(qslEntryStrings.at(0), QString::fromUtf8("hh:mm:ss"));
            }
        oSample.m_dfWorm = (qslEntryStrings.size() > 1)? qslEntryStrings.at(1).toDouble(): 0.;
        oSample.m_dfRa = (qslEntryStrings.size() > 2)? qslEntryStrings.at(2).toDouble(): 0.;
        oSample.m_dfDec = (qslEntryStrings.size() > 3)? qslEntryStrings.at(3).toDouble(): 0.;
        m_lstPeRaw.append(oSample);
        }

    m_bInModelRefill = true;
    int nFirstRow = m_pPeModel->rowCount();
    int nLastRow = m_lstPeRaw.size();
    m_pPeModel->setRowCount(nLastRow);
    for (int nRow = nFirstRow; nRow < nLastRow; nRow++)
        {
        QStandardItem *pItem;
        pItem = new QStandardItem(m_lstPeRaw.at(nRow).m_oTime.toString(QString::fromUtf8("hh:mm:ss.zzz")));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPeModel->setItem(nRow, 0, pItem);
        pItem = new QStandardItem(QString::number(m_lstPeRaw.at(nRow).m_dfWorm, 'f', 3));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPeModel->setItem(nRow, 1, pItem);
        pItem = new QStandardItem(QString::number(m_lstPeRaw.at(nRow).m_dfRa));
        pItem->setEditable(true);
        pItem->setSelectable(true);
        m_pPeModel->setItem(nRow, 2, pItem);
        pItem = new QStandardItem(QString::number(m_lstPeRaw.at(nRow).m_dfDec));
        pItem->setEditable(true);
        pItem->setSelectable(true);
        m_pPeModel->setItem(nRow, 3, pItem);
        }
    m_bInModelRefill = false;

    emit dataPeChanged(this);
    }


QList<PeRawSample> DataPeMcu::samples() const
    {
    return m_lstPeRaw;
    }


void DataPeMcu::changePeItem(QStandardItem *pItem)
    {
    LOG(5, "DataPeMcu::changePeItem");

    if ((pItem == 0) || m_bInModelRefill)
        return;
    int nCol = pItem->column();
    if ((nCol != 2) && (nCol != 3))
        return;
    int nRow = pItem->row();
    if ((nRow < 0) || (nRow >= m_lstPeRaw.size()))
        return;
    double dfValue = pItem->text().toDouble();
    if (nCol == 2)
        {
        if (m_lstPeRaw.at(nRow).m_dfRa == dfValue)
            return;
        m_lstPeRaw[nRow].m_dfRa = dfValue;
        }
    else
        {
        if (m_lstPeRaw.at(nRow).m_dfDec == dfValue)
            return;
        m_lstPeRaw[nRow].m_dfDec = dfValue;
        }

    emit dataPeChanged(this);
    }


