
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2006-2007 by Wolfgang Hoffmann <woho@woho.de>           *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/


#include "datapeiris.h"
#include "ui_datapeiris.h"
#include "log.h"

#include <QtGui>
#include <math.h>


DataPeIris::DataPeIris(QWidget *pParent)
:   QWidget(pParent)
    {
    m_pUi = new Ui::DataPeIris;
    m_pUi->setupUi(this);

    m_pPeModel = new QStandardItemModel(this);
    m_pUi->viewPe->setModel(m_pPeModel);

    connect(m_pPeModel, SIGNAL(itemChanged(QStandardItem *)), this, SLOT(changePeItem(QStandardItem *)));
    }


DataPeIris::~DataPeIris()
    {
    delete m_pUi;
    }


void DataPeIris::clearData()
    {
    m_pPeModel->clear();
    m_lstPeIris.clear();
    m_lstPeRaw.clear();
    }


bool DataPeIris::probePeFile(const QString &qsFileName) const
    {
    LOG(4, "DataPeIris::probePeFile(%s)", LQS(qsFileName));

    bool bIsIrisFormat = false;
    QFile oPeFile(qsFileName);
    if (oPeFile.open(QIODevice::ReadOnly | QIODevice::Text))
        {
        QTextStream oStream(&oPeFile);
        if (!oStream.atEnd())
            {
            QString qsLine = oStream.readLine();
            LOG(5, "    %s", LQS(qsLine));
            QStringList qslEntryStrings = qsLine.split(" ");
            if (qslEntryStrings.size() == 2)
                bIsIrisFormat = true;
            }
        oPeFile.close();
        }

    return bIsIrisFormat;
    }


void DataPeIris::setPeFileName(const QString &qsFileName)
    {
    m_oPeFile.setFileName(qsFileName);
    }


bool DataPeIris::loadPeFile()
    {
    LOG(4, "DataPeIris::loadPeFile(%s)", LQS(m_oPeFile.fileName()));

    clearData();

    m_oPeFile.open(QIODevice::ReadOnly | QIODevice::Text);
    QTextStream oStream(&m_oPeFile);
    while (!oStream.atEnd())
        {
        QString qsLine = oStream.readLine();
        LOG(5, "    %s", LQS(qsLine));
        QStringList qslEntryStrings = qsLine.split(" ");
        PeIrisSample oSampleIris;
        oSampleIris.m_dfTime = (qslEntryStrings.size() > 0)? qslEntryStrings.at(0).toDouble(): -1.;
        oSampleIris.m_dfStarPos = (qslEntryStrings.size() > 1)? qslEntryStrings.at(1).toDouble(): 0.;
        m_lstPeIris.append(oSampleIris);
        PeRawSample oSample;
        oSample.m_oTime = QTime(0, 0).addMSecs((int)(oSampleIris.m_dfTime * 1000.));
        oSample.m_dfWorm = 0.;
        oSample.m_dfRa = 0.;
        oSample.m_dfDec = 0.;
        m_lstPeRaw.append(oSample);
        }
    m_oPeFile.close();

    m_pPeModel->setColumnCount(4);
    m_pPeModel->setRowCount(m_lstPeIris.size());

    QStringList qslHeaders;
    qslHeaders << tr("Time");
    qslHeaders << tr("Star Position");
    qslHeaders << tr("Worm Position");
    qslHeaders << tr("PE");
    m_pPeModel->setHorizontalHeaderLabels(qslHeaders);

    for (int nRow = 0; nRow < m_lstPeIris.size(); nRow++)
        {
        QStandardItem *pItem;
        pItem = new QStandardItem(QString::number(m_lstPeIris.at(nRow).m_dfTime, 'f', 3));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPeModel->setItem(nRow, 0, pItem);
        pItem = new QStandardItem(QString::number(m_lstPeIris.at(nRow).m_dfStarPos));
        pItem->setEditable(true);
        pItem->setSelectable(true);
        m_pPeModel->setItem(nRow, 1, pItem);
        }

    recalcPeRaw();

    return true;
    }


QList<PeRawSample> DataPeIris::samples() const
    {
    return m_lstPeRaw;
    }


void DataPeIris::recalcPeRaw()
    {
    LOG(4, "DataPeIris::recalcPeRaw");
    double dfWormGearPeriod =
        (m_pUi->wormSpeed->currentIndex() == 0)? 86164.: 86400.;    // 0=Sidereal, 1=Solar
    double dfWormDirection =
        ((m_pUi->wormHemisphere->currentIndex() == 0)? -1.: 1.) *   // 0=Northern, 1=Southern
        ((m_pUi->wormRotation->currentIndex() == 0)? 1.: -1.);      // 0=Normal, 1=Reversed
    int nWormGearTeeth = m_pUi->wormGearTeeth->value();
    double dfTime = m_pUi->wormTime->value();
    double dfPos = m_pUi->wormPos->value();
    double dfWormPeriod = dfWormGearPeriod / (double)nWormGearTeeth;
    double dfWormSpeed = 256. / dfWormPeriod * dfWormDirection;
    double dfWormStart = dfPos - dfTime * dfWormSpeed;
    dfWormStart -= floor(dfWormStart / 256.) * 256.;
    double dfScale = m_pUi->aquiPixelScale->value();

    for (int nRow = 0; nRow < m_lstPeIris.size(); nRow++)
        {
        dfTime = m_lstPeIris.at(nRow).m_dfTime;
        double dfWorm = dfWormStart + dfTime * dfWormSpeed;
        dfWorm -= floor(dfWorm / 256.) * 256.;
        double dfPe = (m_lstPeIris.at(nRow).m_dfStarPos - m_lstPeIris.at(0).m_dfStarPos) * dfScale;
        m_lstPeRaw[nRow].m_dfWorm = dfWorm;
        m_lstPeRaw[nRow].m_dfRa = dfPe;
        m_lstPeRaw[nRow].m_dfDec = 0.;
        QStandardItem *pItem;
        pItem = new QStandardItem(QString::number(floor(dfWorm), 'f', 0));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPeModel->setItem(nRow, 2, pItem);
        pItem = new QStandardItem(QString::number(dfPe));
        pItem->setEditable(false);
        pItem->setSelectable(false);
        m_pPeModel->setItem(nRow, 3, pItem);
        }

    emit dataPeChanged(this);
    }


void DataPeIris::setPixelScale(double dfScale)
    {
    m_pUi->aquiPixelScale->setValue(dfScale);
    }


void DataPeIris::setWormSync(double dfTime, int nWormPos)
    {
    m_pUi->wormTime->setValue(dfTime);
    m_pUi->wormPos->setValue(nWormPos);
    }


void DataPeIris::changePeItem(QStandardItem *pItem)
    {
    LOG(5, "DataPeIris::changePeItem");

    if (pItem == 0)
        return;
    int nCol = pItem->column();
    if (nCol != 1)
        return;
    int nRow = pItem->row();
    if ((nRow < 0) || (nRow >= m_lstPeIris.size()))
        return;
    double dfValue = pItem->text().toDouble();
    if (m_lstPeIris.at(nRow).m_dfStarPos == dfValue)
        return;
    m_lstPeIris[nRow].m_dfStarPos = dfValue;

    recalcPeRaw();
    }


void DataPeIris::on_aquiPixelScale_valueChanged(double dfScale)
    {
    LOG(4, "DataPeIris::on_aquiPixelScale_valueChanged");
    Q_UNUSED(dfScale)
    recalcPeRaw();
    }


void DataPeIris::on_wormTime_valueChanged(double dfTime)
    {
    LOG(4, "DataPeIris::on_wormTime_valueChanged");
    Q_UNUSED(dfTime)
    recalcPeRaw();
    }


void DataPeIris::on_wormPos_valueChanged(int nPosition)
    {
    LOG(4, "DataPeIris::on_wormPos_valueChanged");
    Q_UNUSED(nPosition)
    recalcPeRaw();
    }


void DataPeIris::on_wormSpeed_activated(int nSpeed)
    {
    LOG(4, "DataPeIris::on_wormSpeed_activated");
    Q_UNUSED(nSpeed)
    recalcPeRaw();
    }


void DataPeIris::on_wormHemisphere_activated(int nHemisphere)
    {
    LOG(4, "DataPeIris::on_wormHemisphere_activated");
    Q_UNUSED(nHemisphere)
    recalcPeRaw();
    }


void DataPeIris::on_wormRotation_activated(int nRotation)
    {
    LOG(4, "DataPeIris::on_wormRotation_activated");
    Q_UNUSED(nRotation)
    recalcPeRaw();
    }


void DataPeIris::on_wormGearTeeth_valueChanged(int nTeeth)
    {
    LOG(4, "DataPeIris::on_wormGearTeeth_valueChanged");
    Q_UNUSED(nTeeth)
    recalcPeRaw();
    }


