
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2007 by Wolfgang Hoffmann <woho@woho.de>                *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/

#include "data.h"
#include "log.h"

#include <QtCore>
#include <math.h>




/********************************************************************/
/*!
    \class PecSample
    \brief Data container for one PEC table entry
    \sa PecData
*/

/*!
    \enum PecSample::Val
    Possible values for one PEC table entry
*/


int PecSample::val2Int(PecSample::Val eVal)
    {
    switch (eVal)
        {
        default:                return 0;
        case PecSample::faster: return 1;
        case PecSample::slower: return 2;
        case PecSample::reset:  return 3;
        }
    }


PecSample::Val PecSample::int2Val(int nVal)
    {
    switch (nVal)
        {
        default: return PecSample::keep;
        case 1:  return PecSample::faster;
        case 2:  return PecSample::slower;
        case 3:  return PecSample::reset;
        }
    }


int PecSample::val2Acc(PecSample::Val eVal)
    {
    switch (eVal)
        {
        default:                return  0;
        case PecSample::faster: return  1;
        case PecSample::slower: return -1;
        }
    }


PecSample::Val PecSample::acc2Val(int nAcc)
    {
    switch (nAcc)
        {
        default: return PecSample::keep;
        case  1: return PecSample::faster;
        case -1: return PecSample::slower;
        }
    }


PecSample::PecSample()
:   m_eVal(PecSample::reset),
    m_nAcc(0),
    m_nSpd(0),
    m_nPos(0)
    {
    }


PecSample::PecSample(PecSample::Val eVal)
:   m_eVal(eVal),
    m_nAcc(0),
    m_nSpd(0),
    m_nPos(0)
    {
    }


PecSample::PecSample(PecSample::Val eVal, int nAcc, int nSpd, int nPos)
:   m_eVal(eVal),
    m_nAcc(nAcc),
    m_nSpd(nSpd),
    m_nPos(nPos)
    {
    }




/********************************************************************/
/*!
    \class PecNode
    \internal
*/

class PecLevel;


class PecNode
    {
public:
    PecNode(int nRow, int nSpd, int nPos, double dfPosRef);
    int row() const { return m_nRow; };
    int spd() const { return m_nSpd; };
    int pos() const { return m_nPos; };
    void linkAdd(const PecNode *pPrev) { m_lstPrev.append(pPrev); };
    bool linkDelUnreachable();
    bool linked() const { return (m_lstPrev.size() > 0); };
    void initWeight(bool bOnPath);
    void calcWeight();
    double weight() const { return m_dfWeight; };
    const PecNode *bestPrev() const;
    QString debug() const;
private:
    int m_nRow;
    int m_nSpd;
    int m_nPos;
    double m_dfSquErr;
    QList<const PecNode *> m_lstPrev;
    double m_dfWeight;
    };


/*!
*/

PecNode::PecNode(int nRow, int nSpd, int nPos, double dfPosRef)
:   m_nRow(nRow), m_nSpd(nSpd), m_nPos(nPos)
    {
    double dfError = (double)m_nPos - dfPosRef;
    m_dfSquErr = dfError * dfError;
    }


/*!
    Remove links to all prev nodes that are unreachable, i.e.
    that have no prev nodes themselves

    Return true if at least one link was removed.
*/

bool PecNode::linkDelUnreachable()
    {
    int nPrevNum = m_lstPrev.size();
    for (int nPrev = nPrevNum - 1; nPrev >= 0; nPrev--)
        if (!m_lstPrev[nPrev]->linked())
            m_lstPrev.removeAt(nPrev);
    return (nPrevNum > m_lstPrev.size());
    }


/*!
    Initialise weight to 0 if \a bOnPath is true, 1e10 otherwise.
*/

void PecNode::initWeight(bool bOnPath)
    {
    m_dfWeight = (bOnPath)? 0.: 1e10;
    }


/*!
    Calculate weight of this node by adding the square error
    of this node to the weight of its best previous node
    (best meaning lowest weight).
*/

void PecNode::calcWeight()
    {
    double dfWeightPrev = 1e10;
    for (int nPrev = 0; nPrev < m_lstPrev.size(); nPrev++)
        if (dfWeightPrev > m_lstPrev.at(nPrev)->weight())
            dfWeightPrev = m_lstPrev.at(nPrev)->weight();
    m_dfWeight = m_dfSquErr + dfWeightPrev;
    }


/*!
    Return the best previous node (best meaning lowest weight).
*/

const PecNode *PecNode::bestPrev() const
    {
    const PecNode *pNodePrev = 0;
    double dfWeightPrev = 1e9;
    for (int nPrev = 0; nPrev < m_lstPrev.size(); nPrev++)
        if (dfWeightPrev > m_lstPrev.at(nPrev)->weight())
            {
            dfWeightPrev = m_lstPrev.at(nPrev)->weight();
            pNodePrev = m_lstPrev.at(nPrev);
            }
    return pNodePrev;
    }


QString PecNode::debug() const
    {
    QString qsDebug = QString::fromUtf8("   row %1, pos %2, spd %3:")
        .arg(m_nRow, 3)
        .arg(m_nPos, 3)
        .arg(m_nSpd, 2);
    for (int nPrev = 0; nPrev < m_lstPrev.size(); nPrev++)
        qsDebug += QString::fromUtf8(" (%1%2%3)")
            .arg(m_lstPrev.at(nPrev)->pos())
            .arg((m_lstPrev.at(nPrev)->spd() >= 0)? QString::fromUtf8("+"): QString())
            .arg(m_lstPrev.at(nPrev)->spd());
    return qsDebug;
    }




/********************************************************************/
/*!
    \class PecLevel
    \internal
*/


class PecLevel
    {
public:
    PecLevel() {};
    void init(int nRow, double dfProfile,
        int nAgr, int nSpdMax, int nPosMin, int nPosMax)
        {
        m_nAgr = nAgr;
        m_nSpdMax = nSpdMax;
        m_nPosMin = nPosMin;
        m_nPosMax = nPosMax;
        for (int nPos = nPosMin; nPos <= nPosMax; nPos++)
            for (int nSpd = -nSpdMax; nSpd <= nSpdMax; nSpd += nAgr)
                m_lstPecPoint.append(PecNode(nRow, nSpd, nPos, dfProfile));
        };
    int agr() const { return m_nAgr; };
    int spdMax() const { return m_nSpdMax; };
    int size() const { return m_lstPecPoint.size(); };
    const PecNode &at(int nNode) const { return m_lstPecPoint.at(nNode); };
    PecNode &operator[](int nNode) { return m_lstPecPoint[nNode]; };
    const PecNode *getNode(int nPos, int nSpd) const;
    void linkAddReachable(const PecLevel &oLevelPrev);
    bool linkDelUnreachable();
    void removeUnreachable();
private:
    int m_nAgr;
    int m_nSpdMax;
    int m_nPosMin;
    int m_nPosMax;
    QList<PecNode> m_lstPecPoint;
    };


/*!
    Get node with position \a nPos and speed \a nSpd

    \note This must not be called after the first call to removeUnreachable(),
        as it relies on a fixed layout of the nodes in the list. Removing
        nodes from the list changes that layout.
*/
const PecNode *PecLevel::getNode(int nPos, int nSpd) const
    {
    if ((nSpd < -m_nSpdMax) || (nSpd > m_nSpdMax) || ((nSpd % m_nAgr) != 0))
        return 0;
    int nNode = (nPos - m_nPosMin) * (m_nSpdMax / m_nAgr * 2 + 1) + (nSpd + m_nSpdMax) / m_nAgr;
    if ((nNode < 0) || (nNode >= m_lstPecPoint.size()))
        return 0;
    return &(m_lstPecPoint.at(nNode));
    }


/*!
    For each node in this level, link this node with all nodes
    from previous level \a pLevelPrev that can reach this node, meaning:
    - this position is reached when moving from previous position
      with previous speed
    - this speed can be reached by PEC table means, i.e. this speed
      is 0 (PecSample::reset) or at previous speed + aggressiveness * {-1,0,1}
      (PecSample::slower, PecSample::keep, PecSample::faster)
*/

void PecLevel::linkAddReachable(const PecLevel &oLevelPrev)
    {
    for (int nNode = 0; nNode < m_lstPecPoint.size(); nNode++)
        {
        PecNode &rThis = m_lstPecPoint[nNode];
        int nPosThis = rThis.pos();
        int nSpdThis = rThis.spd();
        int nSpdPrevMin = (nSpdThis == 0)? -m_nSpdMax: nSpdThis - m_nAgr;
        int nSpdPrevMax = (nSpdThis == 0)?  m_nSpdMax: nSpdThis + m_nAgr;
        for (int nSpdPrev = nSpdPrevMin; nSpdPrev <= nSpdPrevMax; nSpdPrev += m_nAgr)
            {
            int nPosPrev = nPosThis - nSpdPrev;
            const PecNode *pNode = oLevelPrev.getNode(nPosPrev, nSpdPrev);
            if (pNode != 0)
                rThis.linkAdd(pNode);
            }
        }
    }


/*!
    Call PecNode::linkDelUnreachable() for each node in this level

    Returns true if at least one link was removed.
*/

bool PecLevel::linkDelUnreachable()
    {
    bool bDone = true;
    for (int nNode = 0; nNode < m_lstPecPoint.size(); nNode++)
        if (m_lstPecPoint[nNode].linkDelUnreachable())
            bDone = false;
    return bDone;
    }


/*!
    Remove all nodes from this level that are not linked to any
    nodes from the previous level.
*/

void PecLevel::removeUnreachable()
    {
    // careful: iterate backwards, as we're potentially deleting items
    for (int nNode = m_lstPecPoint.size() - 1; nNode >= 0; nNode--)
        if (!m_lstPecPoint.at(nNode).linked())
            m_lstPecPoint.removeAt(nNode);
    }




/********************************************************************/
/*!
    \class PecData
    \brief Data container for PEC table

    This class holds data for PEC (periodic error correction).
*/


PecData::PecData()
    {
    m_aTable.resize(c_nRowNum);
    clear();
    }


/*!
    Clear PEC table

    Sets all PEC value to PecSample::reset.
*/

void PecData::clear()
    {
    m_eTraversal = backward;
    m_nFactor = 1;
    m_nAggress = 0;
    for (int nRow = 0; nRow < c_nRowNum; nRow++)
        m_aTable[nRow] = PecSample();
    m_aTable[0].m_eVal = PecSample::reset;
    m_nOffset = 0;
    }


PecData::Traversal PecData::traversal() const
    {
    return m_eTraversal;
    }


void PecData::setTraversal(Traversal eDirection)
    {
    m_eTraversal = eDirection;
    recalcGeo();
    }


int PecData::factor() const
    {
    return m_nFactor;
    }


int PecData::aggress() const
    {
    return m_nAggress;
    }


int PecData::fileFactorAggress() const
    {
    return ((m_nAggress & 0x0f) << 4) | (m_nFactor & 0x0f);
    }


const PecSample &PecData::operator[](int nRow) const
    {
    static PecSample oNull;
    if ((nRow < 0) || (nRow >= c_nRowNum))
        return oNull;

    return m_aTable[nRow];
    }


PecSample::Val PecData::value(int nRow) const
    {
    return operator[](nRow).m_eVal;
    }


int PecData::valueInt(int nRow) const
    {
    return PecSample::val2Int(value(nRow));
    }


bool PecData::set(int nFactor, int nAggress, const QVector<PecSample::Val> &aTable)
    {
    if (aTable.size() != c_nRowNum)
        return false;

    m_nFactor = nFactor;
    m_nAggress = nAggress;
    for (int nRow = 0; nRow < c_nRowNum; nRow++)
        m_aTable[nRow].m_eVal = aTable[nRow];

    recalcGeo();
    return true;
    }


bool PecData::setFactor(int nFactor)
    {
    m_nFactor = nFactor;
    return true;
    }


bool PecData::setAggress(int nAggress)
    {
    m_nAggress = nAggress;
    return true;
    }


bool PecData::setFileFactorAggress(int nFactorAggress)
    {
    m_nFactor = (nFactorAggress & 0x0f);
    m_nAggress = ((nFactorAggress >> 4) & 0x0f);
    return true;
    }


bool PecData::setValue(int nRow, PecSample::Val eVal)
    {
    if ((nRow < 0) || (nRow >= c_nRowNum))
        return false;

    m_aTable[nRow].m_eVal = eVal;
    return true;
    }


void PecData::setFinish()
    {
    recalcGeo();
    }


/*!
    Calculate PEC table to achieve the given PE profile \a aProfile

    Background:

    PE is a positional error of the worm gear.

    Let \c x denote the rotation angle of the worm, running from \c 0 ... \c 256
    for one worm revolution (360°).<br>
    Let \c y denote the rotation angle of the worm wheel, running from \c 0 ... \c 256
    when advancing by one worm wheel tooth (360° / number of worm wheel teeth).

    Ideally, <tt>y = x + C</tt>, with C being an arbitrary constant
    (worm and worm wheel beeing endless, so there's no special start point).<br>
    With PE, <tt>y = x + e(x) + C</tt>.

    To compensate the PE, the MCU models e(x) with an approximation c(x), which has
    discrete slope values. The MCU holds a PEC table of 256 entries and a PEC factor f,
    which describe <tt>c(x)</tt>. A PEC table entry has one of the following values:
    - \c 0: slope of c(x) is unchanged,<br>
        i.e. <tt>c(x+1) - c(x) = c(x) - c(x-1)</tt>
    - \c 1: slope of c(x) increases by a, if slope is \< f, otherwise is unchanged,<br>
        i.e. <tt>c(x+1) - c(x) = c(x) - c(x-1) + a</tt>
    - \c 2: slope of c(x) decreases by a, if slope is \> -f, otherwise is unchanged,<br>
        i.e. <tt>c(x+1) - c(x) = c(x) - c(x-1) - a</tt>
    - \c 3: slope of c(x) is set to 0,<br>
        i.e. <tt>c(x+1) - c(x) = 0</tt>

    c(x) is dependent on the traversal direction, i.e. whether the
    PEC table is processed with increasing or decreasing x.
    When traversing backwards, the MCU uses the PEC table entry at x to
    calculate the speed while moving from c(x+1) to c(x), i.e. it does a kind
    of look-ahead.
    It seems that currently the MCU always traverses the PEC table with
    decreasing x, so switching the RA direction (from Normal to Reversed or vice versa)
    or switching the Hemisphere invalidates the PEC table.

    Concerning \c a, empirical measurements suggest a value of 1/80 for firmware v3.59.
    With firmware v4.00, a PEC aggressiveness is introduced, making \c a configurable:
    \c a = 1/240 * aggressiveness

    \sa recalcGeo()
*/

bool PecData::calc(const QVector<double> &aProfile, double dfPecPosScale,
    int nAggressMin, int nAggressMax, int nFactorMax, IProgress *pProgress)
    {
    LOG(4, "PecData::calc(): traversal %d", (int)m_eTraversal);
    if (aProfile.size() != c_nRowNum)
        {
        LOG(0, "PecData::calc(): aProfile has wrong size %d instead of %d",
            aProfile.size(), c_nRowNum);
        return false;
        }

    if (pProgress != 0)
        pProgress->setValue(0.);

    // work on local PEC table until finished, to preserve existing table
    // if the user cancels calculation
    int nResultOffset = 0;
    int nResultFactor = 1;
    int nResultAggress = (nAggressMin < 1)? 0: 1;
    QVector<PecSample> aResultTable;
    aResultTable.resize(c_nRowNum);
    for (int nRow = 0; nRow < c_nRowNum; nRow++)
        aResultTable[nRow] = PecSample::reset;

    // make sure limits are sane; take firmware < 4 into account that lacks
    // aggressiveness parameter by mapping 0 to 1
    if (nAggressMin < 1)
        nAggressMin = 1;
    if (nAggressMin > c_nAggressNum)
        nAggressMin = c_nAggressNum;
    if (nAggressMax < nAggressMin)
        nAggressMax = nAggressMin;
    if (nAggressMax > c_nAggressNum)
        nAggressMax = c_nAggressNum;
    if (nFactorMax > c_nFactorNum)
        nFactorMax = c_nFactorNum;

    // estimate calculation time for each aggressiveness to get a smooth
    // progress indicator
    double adfProgress[nAggressMax - nAggressMin + 2];
    adfProgress[0] = 0.;
    double dfProgressSum = 0.;
    for (int nAgr = nAggressMin; nAgr <= nAggressMax; nAgr++)
        {
        dfProgressSum += (double)nFactorMax * (double)((nFactorMax - 1) / nAgr + 1);
        adfProgress[nAgr - nAggressMin + 1] = dfProgressSum;
        }
    for (int nAgr = nAggressMin; nAgr <= nAggressMax; nAgr++)
        adfProgress[nAgr - nAggressMin + 1] /= dfProgressSum;

    // Find PEC table with minimal square error between ideal and resulting
    // PEC curve. To do so, use a semi-brute-force approach ...
    double dfWeightTotMin = 1e9;
    bool bSuccess = false;
    for (int nAgr = nAggressMin; nAgr <= nAggressMax; nAgr++)
        {
        LOG(4, "    aggressiveness %d:", nAgr);

        // this might take a while, so do proper progress indication
        // flooding the graph is what takes longest
        double dfProgressBeg = adfProgress[nAgr - nAggressMin];
        double dfProgressEnd = adfProgress[nAgr - nAggressMin + 1];
        double dfProgressGraphBeg = dfProgressBeg + (dfProgressEnd - dfProgressBeg) * 0.00;
        double dfProgressGraphEnd = dfProgressBeg + (dfProgressEnd - dfProgressBeg) * 0.03;
        double dfProgressNodesBeg = dfProgressBeg + (dfProgressEnd - dfProgressBeg) * 0.03;
        double dfProgressNodesEnd = dfProgressBeg + (dfProgressEnd - dfProgressBeg) * 0.08;
        double dfProgressReduceBeg = dfProgressBeg + (dfProgressEnd - dfProgressBeg) * 0.08;
        double dfProgressReduceEnd = dfProgressBeg + (dfProgressEnd - dfProgressBeg) * 0.10;
        double dfProgressFloodBeg = dfProgressBeg + (dfProgressEnd - dfProgressBeg) * 0.10;
        double dfProgressFloodEnd = dfProgressBeg + (dfProgressEnd - dfProgressBeg) * 0.99;
        double dfProgressPecBeg = dfProgressBeg + (dfProgressEnd - dfProgressBeg) * 0.99;
        double dfProgressPecEnd = dfProgressBeg + (dfProgressEnd - dfProgressBeg) * 1.00;

        // A node is a PEC (point,position,speed) triplet.
        // Make a directed graph of these nodes, with PEC point as level criterion.
        // Graph direction is PEC traversal direction, i.e. level increments as
        // PEC table is traversed.
        // Hold the graph in an array of PEC point node lists.
        PecLevel aPecLevel[c_nRowNum];

        // Populate graph with nodes representing all positions within
        // profile range +/- nFactorMax vertically and +/- c_nRowGen horizontally,
        // all aggressiveness values and all speeds that are multiple of
        // aggressiveness and within +/- (nFactorMax - 1 + nAggressMax)
        LOG(4, "        populating graph ...");
        int nNodeTotNum = 0;
        for (int nLevel = 0; nLevel < c_nRowNum; nLevel++)
            {
            if (pProgress != 0)
                if (pProgress->setValue(dfProgressGraphBeg +
                    (dfProgressGraphEnd - dfProgressGraphBeg) * (double)nLevel / (double)(c_nRowNum - 1)))
                    {
                    LOG(1, "    agr %d: populating graph ...", nAgr);
                    LOG(1, "        ... canceled by user");
                    return false;
                    }
            int nRowThis = (m_eTraversal * nLevel + c_nRowNum) % c_nRowNum;
            double dfProfileMin = aProfile[nRowThis] / dfPecPosScale;
            double dfProfileMax = dfProfileMin;
            for (int nOffs = -c_nRowGen; nOffs <= c_nRowGen; nOffs++)
                {
                int nRowNext = (m_eTraversal * (nLevel + nOffs) + 2 * c_nRowNum) % c_nRowNum;
                if (dfProfileMin > aProfile[nRowNext] / dfPecPosScale)
                    dfProfileMin = aProfile[nRowNext] / dfPecPosScale;
                if (dfProfileMax < aProfile[nRowNext] / dfPecPosScale)
                    dfProfileMax = aProfile[nRowNext] / dfPecPosScale;
                }
            int nProfileMin = (int)floor(dfProfileMin) - nFactorMax;
            int nProfileMax = (int)ceil(dfProfileMax) + nFactorMax;
            int nSpdMax = ((nFactorMax - 1) / nAgr + 1) * nAgr;
            aPecLevel[nLevel].init(nRowThis, aProfile[nRowThis] / dfPecPosScale,
                nAgr, nSpdMax, nProfileMin, nProfileMax);
            nNodeTotNum += aPecLevel[nLevel].size();
            }

        // Connect each node to all nodes of prev level that are within reach of
        // one PEC step (given the current and prev position and speed as well as
        // the 4 possible PEC table accelerations and the aggressiveness)
        for (int nLevelThis = 0; nLevelThis < c_nRowNum; nLevelThis++)
            {
            if (pProgress != 0)
                if (pProgress->setValue(dfProgressNodesBeg +
                    (dfProgressNodesEnd - dfProgressNodesBeg) * (double)nLevelThis / (double)c_nRowNum))
                    {
                    LOG(1, "    agr %d: linking nodes ...", nAgr);
                    LOG(1, "        ... canceled by user");
                    return false;
                    }
            int nLevelPrev = (nLevelThis - 1 + c_nRowNum) % c_nRowNum;
            aPecLevel[nLevelThis].linkAddReachable(aPecLevel[nLevelPrev]);
            }

        // Remove all unreachable nodes, i.e. nodes that are not linked to
        // the previous level. Do this once for each level and then continue
        // until reaching a level where no further nodes can be removed
        for (int nLevel = 0; ; nLevel++)
            {
            if (pProgress != 0)
                if (pProgress->setValue(dfProgressReduceBeg +
                    (dfProgressReduceEnd - dfProgressReduceBeg) * (double)nLevel * .5 / (double)c_nRowNum))
                    {
                    LOG(1, "    agr %d: removing unreachable nodes ...", nAgr);
                    LOG(1, "        ... canceled by user");
                    return false;
                    }
            // first unlink all unreachable nodes of previous level
            int nLevelThis = nLevel % c_nRowNum;
            bool bDone = aPecLevel[nLevelThis].linkDelUnreachable();
            // then remove these nodes from graph
            int nLevelPrev = (nLevelThis - 1 + c_nRowNum) % c_nRowNum;
            aPecLevel[nLevelPrev].removeUnreachable();
            if ((nLevel >= c_nRowNum) && bDone)
                break;
            }
        int nNodeLnkNum = 0;
        for (int nLevel = 0; nLevel < c_nRowNum; nLevel++)
            for (int nNode = 0; nNode < aPecLevel[nLevel].size(); nNode++)
                if (aPecLevel[nLevel].at(nNode).linked())
                    nNodeLnkNum += 1;
        LOG(4, "            ... done: %d nodes, %d linked", nNodeTotNum, nNodeLnkNum);

        // Find level with the least number of nodes as startpoint of graph
        // flooding and split cyclic graph there
        int nLevelSplit = 0;
        for (int nLevel = 1; nLevel < c_nRowNum; nLevel++)
            if (aPecLevel[nLevelSplit].size() > aPecLevel[nLevel].size())
                nLevelSplit = nLevel;
        PecLevel &rLevelSplit = aPecLevel[nLevelSplit];

        // catch cornercase of PE curve that has so extreme values
        // that all nodes of one level are not reachable
        if (rLevelSplit.size() == 0)
            {
            LOG(1, "    agr %d: creating PEC table ...", nAgr);
            LOG(1, "        ... failed: graph is not traversable.");
            continue;
            }

        // Now that all possible connections are known, find best route, i.e.
        // route that minimizes square error between ideal and resulting PEC curve.
        // To do so, for each bottom node flood graph to calculate error weights
        // of best route from this bottom node to it's corresponding top node.
        // Note: it's not enough to flood once and find best overall route, as
        // only routes starting and ending in the same node are valid
        LOG(4, "        flood graph, starting from level %d (%d nodes) ...",
            nLevelSplit, rLevelSplit.size());
        double dfWeightMin = 1e10;
        int nNodeMin = 0;
        for (int nNodeCnt = 0; nNodeCnt < rLevelSplit.size() + 1; nNodeCnt++)
            {
            if (pProgress != 0)
                if (pProgress->setValue(dfProgressFloodBeg +
                    (dfProgressFloodEnd - dfProgressFloodBeg) * (double)nNodeCnt / (double)rLevelSplit.size()))
                    {
                    LOG(1, "    agr %d: flooding graph ...", nAgr);
                    LOG(1, "        ... canceled by user");
                    return false;
                    }
            // trick: loop over all potential start nodes to find best one;
            // then loop one more time to have graph weights valid for best node
            int nNodeStart = (nNodeCnt < rLevelSplit.size())? nNodeCnt: nNodeMin;
            for (int nNodeThis = 0; nNodeThis < rLevelSplit.size(); nNodeThis++)
                rLevelSplit[nNodeThis].initWeight(nNodeThis == nNodeStart);
            for (int nLevel = nLevelSplit + 1; nLevel <= nLevelSplit + c_nRowNum; nLevel++)
                {
                int nLevelThis = nLevel % c_nRowNum;
                for (int nNodeThis = 0; nNodeThis < aPecLevel[nLevelThis].size(); nNodeThis++)
                    aPecLevel[nLevelThis][nNodeThis].calcWeight();
                }
            LOG(5, "    node %d (pos %d, spd %d): %f", nNodeCnt,
                rLevelSplit.at(nNodeStart).pos(),
                rLevelSplit.at(nNodeStart).spd(),
                rLevelSplit.at(nNodeStart).weight());
            if (dfWeightMin > rLevelSplit.at(nNodeStart).weight())
                {
                dfWeightMin = rLevelSplit.at(nNodeStart).weight();
                nNodeMin = nNodeStart;
                }
            }
        LOG(4, "            ... done: best node %d, weight %f", nNodeMin, dfWeightMin);

        // Do a consistency check before continuing ...
        const PecNode *pNode = &(rLevelSplit.at(nNodeMin));
        for (int nCnt = 0; nCnt < c_nRowNum; nCnt++)
            if (pNode != 0)
                pNode = pNode->bestPrev();
        if (pNode == 0)
            {
            LOG(1, "    agr %d: creating PEC table ...", nAgr);
            LOG(1, "        ... failed: no route through graph found.");
            continue;
            }
        LOG(1, "    agr %d: MSE %0.3f\" (graph: tot %d (%d), lvl %d (%d), best %d)",
            nAgr, dfWeightMin / (double)c_nRowNum * dfPecPosScale * dfPecPosScale,
            nNodeTotNum, nNodeLnkNum, nLevelSplit, rLevelSplit.size(), nNodeMin);

        // Discard results of this aggressiveness if no path through PEC profile
        // found, or if a previous aggressiveness gave a better result.
        if (dfWeightTotMin <= dfWeightMin)
            continue;
        dfWeightTotMin = dfWeightMin;
        bSuccess = true;

        // Otherwise fill PEC table according to best route through graph.
        if (pProgress != 0)
            if (pProgress->setValue(dfProgressPecBeg))
                {
                LOG(1, "    agr %d: creating PEC table ...", nAgr);
                LOG(1, "        ... canceled by user");
                return false;
                }
        int nPosMin = 1000;
        int nPosMax = -1000;
        int nSpdMax = 1;
        pNode = &(rLevelSplit.at(nNodeMin));
        for (int nCnt = 0; nCnt < c_nRowNum; nCnt++)
            {
            const PecNode *pNodePrev = pNode->bestPrev();
            // The firmware seems to model a PEC table entry not as a point,
            // but as an interval. The interval covers the area of the worm
            // where pecByte*4+pecBit is constant.
            // The PEC table entry always needs to be placed at the left side
            // of the intervall, so we need to distinguish traveling direction here
            int nRow = (m_eTraversal == forward)? pNode->row():
                ((pNode->row() - 1 + c_nRowNum) % c_nRowNum);
            int nPos = pNode->pos();
            int nSpd = pNode->spd();
            int nAcc = nSpd - pNodePrev->spd();
            if (nSpd == 0)
                aResultTable[nRow] = PecSample::reset;
            else
                aResultTable[nRow] = PecSample::acc2Val(nAcc / nAgr);
            if (nSpdMax < abs(nSpd))
                nSpdMax = abs(nSpd);
            if (nPosMin > nPos)
                nPosMin = nPos;
            if (nPosMax < nPos)
                nPosMax = nPos;
            pNode = pNodePrev;
            }
        nResultAggress = (nResultAggress < 1)? 0: nAgr;
        nResultFactor = (nSpdMax < c_nFactorNum)? nSpdMax: c_nFactorNum;
        nResultOffset = (nPosMax < nPosMin)? 0: ((nPosMax + nPosMin) / 2);
        if (pProgress != 0)
            pProgress->setValue(dfProgressPecEnd);
        }

    if (bSuccess)
        {
        // Fill in speed and position profile from the generated acceleration sequence
        m_nOffset = nResultOffset;
        m_nFactor = nResultFactor;
        m_nAggress = nResultAggress;
        for (int nRow = 0; nRow < c_nRowNum; nRow++)
            m_aTable[nRow] = aResultTable[nRow];
        recalcGeo();
        }

    return bSuccess;
    }


/*!
    Recalculate acceleration, speed and position from
    raw PEC table values
*/

void PecData::recalcGeo()
    {
    // traverse twice through the table:
    // - first time to get a correct start speed
    //   necessary for tables without any PecSample::reset value;
    //   still this might not work out, consider e.g. a table of 0's only
    // - second time for storing acc, spd and pos
    int nAgr = (m_nAggress > 0)? m_nAggress: 1;
    int nSpd = 0;
    for (int nRound = 0; nRound < 2; nRound++)
        {
        int nPos = 0;
        for (int nCnt = 0; nCnt < c_nRowNum; nCnt++)
            {
            // careful:
            // The firmware seems to model a PEC table entry not as a point,
            // but as an interval. The interval covers the area of the worm
            // where pecByte*4+pecBit is constant. Physical properties are
            // modeled like this:
            // - acceleration is a spike at the interval entry point, i.e.
            //   it's real position on the worm depends on traversal direction
            // - speed changes immediately at interval entry, since accel is a spike
            //   so speed is the same over the interval, independently of traversal
            //   direction, except if previous speed differs
            // - position reaches resulting value when exiting the interval,
            //   after moving from current to next point with current speed
            // So when looking at a PEC table entry point, it's values depend
            // on traversal direction:
            // - forward (this=n, prev=n-1, next=n+1):
            //   speed[this] = speed[prev] + acc[this]
            //   pos[this] = pos[prev] + speed[prev]
            // - backward (this=n, prev=n+1, next=n-1):
            //   speed[this] = speed[prev] + acc[next]
            //   pos[this] = pos[prev] + speed[prev]

            // nSpd and nPos are speed and position at entry of current interval
            int nRowThis = (m_eTraversal * nCnt + c_nRowNum) % c_nRowNum;
            int nRowNext = (m_eTraversal * (nCnt + 1) + c_nRowNum * 2) % c_nRowNum;
            int nRowAcc = (m_eTraversal == forward)? nRowThis: nRowNext;
            int nAcc = 0;
            switch (m_aTable[nRowAcc].m_eVal)
                {
                case PecSample::reset:  nAcc =  0; nSpd = 0; break;
                case PecSample::faster: nAcc =  1; break;
                case PecSample::slower: nAcc = -1; break;
                case PecSample::keep:   nAcc =  0; break;
                }
            // limit speed according to PEC factor.
            // a speedup is allowed if speed < factor, a slowdown if speed > -factor
            // note that speed can exceed factor if aggressiveness is > 1
            if (((nAcc > 0) && (nSpd <  m_nFactor)) ||
                ((nAcc < 0) && (nSpd > -m_nFactor)))
                nSpd += nAcc * nAgr;
            if (nRound == 0)
                continue;
            PecSample &rThis = m_aTable[nRowThis];
            rThis.m_nAcc = nAcc;
            rThis.m_nSpd = nSpd;
            rThis.m_nPos = nPos;
            nPos += nSpd;
            }
        }

    // center positions such that m_nOffset is at half distance between minimum and maximum
    int nRowMin = 0;
    int nRowMax = 0;
    for (int nRow = 0; nRow < c_nRowNum; nRow++)
        {
        if (m_aTable.at(nRowMin).m_nPos > m_aTable.at(nRow).m_nPos)
            nRowMin = nRow;
        if (m_aTable.at(nRowMax).m_nPos < m_aTable.at(nRow).m_nPos)
            nRowMax = nRow;
        }
    int nPosMid = (m_aTable.at(nRowMin).m_nPos + m_aTable.at(nRowMax).m_nPos) / 2 - m_nOffset;
    for (int nRow = 0; nRow < c_nRowNum; nRow++)
        m_aTable[nRow].m_nPos -= nPosMid;
    }

