
/***************************************************************************
 *   This file is part of Aspect, a simple PEC tool.                       *
 *                                                                         *
 *   Copyright (C) 2006-2007 by Wolfgang Hoffmann <woho@woho.de>           *
 *                                                                         *
 *   This program is free software, licensed under the GPL v2.             *
 *   See the file COPYING for more details.                                *
 ***************************************************************************/


#include "chart.h"
#include "log.h"

#include <QtGui>
#include <math.h>




/********************************************************************/
/*!
    \class Graph
    \brief Graph for visualizing one PE or PEC curve

    \sa Grid, Chart
*/

Graph::Graph(Style eStyle, double dfXBeg, double dfXEnd, double dfYBeg, double dfYEnd)
:   m_eStyle(eStyle),
    m_dfXBeg(dfXBeg),
    m_dfXEnd(dfXEnd),
    m_dfYBeg(dfYBeg),
    m_dfYEnd(dfYEnd),
    m_dfXMin(0.),
    m_dfXMax(0.),
    m_dfYMin(0.),
    m_dfYMax(0.),
    m_bFirst(true)
    {
    }


void Graph::addData(double dfX, double dfY, bool bNewSubPath)
    {
    // prevent numbers from being inf/nan or otherwise insane,
    // to avoid confusing or overloading QGraphicsView
    // this cures the symptoms of the chart area staying blank forever
    // and aspect hogging the CPU while calculating a massive amount
    // of grid-lines
    if (!(dfX > m_dfXBeg))
        dfX = m_dfXBeg;
    if (!(dfX < m_dfXEnd))
        dfX = m_dfXEnd;
    if (!(dfY > m_dfYBeg))
        dfY = m_dfYBeg;
    if (!(dfY < m_dfYEnd))
        dfY = m_dfYEnd;

    if (m_dfXMin > dfX)
        m_dfXMin = dfX;
    if (m_dfXMax < dfX)
        m_dfXMax = dfX;
    if (m_dfYMin > dfY)
        m_dfYMin = dfY;
    if (m_dfYMax < dfY)
        m_dfYMax = dfY;

    if ((!m_bFirst) && (bNewSubPath))
        {
        m_oPath.moveTo(0., 0.);
        m_bFirst = true;
        }

    switch (m_eStyle)
        {
        case dot:
            {
            QRectF oBound(0., 0., 2., 2.);
            oBound.moveCenter(QPointF(dfX, dfY));
            m_oPath.addEllipse(oBound);
            }
            break;
        case tick:
            m_oPath.moveTo(dfX, dfY - .25);
            m_oPath.lineTo(dfX, dfY + .25);
            break;
        case line:
            if (m_bFirst)
                m_oPath.moveTo(dfX, dfY);
            else
                m_oPath.lineTo(dfX, dfY);
            break;
        case stairLeft:
            if (m_bFirst)
                m_oPath.moveTo(dfX, dfY);
            else
                {
                m_oPath.lineTo(dfX, m_oPath.currentPosition().y());
                m_oPath.lineTo(dfX, dfY);
                }
            break;
        case stairRight:
            if (m_bFirst)
                m_oPath.moveTo(dfX, dfY);
            else
                {
                m_oPath.lineTo(m_oPath.currentPosition().x(), dfY);
                m_oPath.lineTo(dfX, dfY);
                }
            break;
        }
    m_bFirst = false;
    }


void Graph::render(QGraphicsScene *pScene,
    double dfXScale, double dfYZero, double dfYScale,
    double dfZValue, const QPen &oPen)
    {
    if (m_oPath.isEmpty())
        return;
    m_oPath.moveTo(0., 0.);

    // graphics view has y incrementing from top to bottom
    // matrix is evaluated backwards, i.e. when applied, translation comes first.
    QMatrix oMatrix;
    oMatrix.scale(dfXScale, -dfYScale);
    oMatrix.translate(0., -dfYZero);

    QGraphicsPathItem *pItem = pScene->addPath(m_oPath, oPen);
    pItem->setZValue(dfZValue);
    pItem->setMatrix(oMatrix, true);
    }




/********************************************************************/
/*!
    \class Grid
    \brief Grid for data chart

    \sa Graph, Chart
*/

Grid::Grid()
:   m_dfXMin(0.),
    m_dfXMax(0.),
    m_dfYMin(0.),
    m_dfYMax(0.)
    {
    }


void Grid::addData(double dfX, double dfY)
    {
    if (m_dfXMin > dfX)
        m_dfXMin = dfX;
    if (m_dfXMax < dfX)
        m_dfXMax = dfX;
    if (m_dfYMin > dfY)
        m_dfYMin = dfY;
    if (m_dfYMax < dfY)
        m_dfYMax = dfY;
    }


void Grid::addGraph(const Graph &oGraph)
    {
    if (m_dfXMin > oGraph.minX())
        m_dfXMin = oGraph.minX();
    if (m_dfXMax < oGraph.maxX())
        m_dfXMax = oGraph.maxX();
    if (m_dfYMin > oGraph.minY())
        m_dfYMin = oGraph.minY();
    if (m_dfYMax < oGraph.maxY())
        m_dfYMax = oGraph.maxY();
    }


double Grid::minYSanitized() const
    {
    double dfYMin = m_dfYMin;
    double dfYRange = m_dfYMax - m_dfYMin;
    if (dfYRange < 10.)
        dfYMin -= (10. - dfYRange) / 2.;
    if (dfYRange > 300.)
        dfYMin += (dfYRange - 300.) / 2.;
    return floor(dfYMin);
    }


double Grid::maxYSanitized() const
    {
    double dfYMax = m_dfYMax;
    double dfYRange = m_dfYMax - m_dfYMin;
    if (dfYRange < 10.)
        dfYMax += (10. - dfYRange) / 2.;
    if (dfYRange > 300.)
        dfYMax -= (dfYRange - 300.) / 2.;
    return ceil(dfYMax);
    }


void Grid::render(QGraphicsScene *pScene,
    double dfXScale, double dfYZero, double dfYScale,
    Lines eLineX, Lines eLineY, Numbers eNumX, Numbers eNumY,
    double dfMajorX, double dfMinorX, double dfMajorY, double dfMinorY,
    double dfMajorZ, double dfMinorZ, const QColor &oNumberColor,
    const QPen &oMajorPen, const QPen &oMinorPen)
    {
    if ((m_dfXMin == 0.) && (m_dfXMax == 0.) && (m_dfYMin == 0.) && (m_dfYMax == 0.))
        return;

    double dfYRange = m_dfYMax - m_dfYMin;
    if ((dfYRange < 10.) && ((eLineY & major) == major))
        {
        m_dfYMin -= (10. - dfYRange) / 2.;
        m_dfYMax += (10. - dfYRange) / 2.;
        }
    if (dfYRange > 300.)
        {
        m_dfYMin += (dfYRange - 300.) / 2.;
        m_dfYMax -= (dfYRange - 300.) / 2.;
        }

    QPainterPath oPathMinor;
    double dfXMinorMin = floor(m_dfXMin / dfMinorX) * dfMinorX;
    double dfXMinorMax = ceil(m_dfXMax / dfMinorX) * dfMinorX;
    double dfYMinorMin = floor(m_dfYMin / dfMinorY) * dfMinorY;
    double dfYMinorMax = ceil(m_dfYMax / dfMinorY) * dfMinorY;
    if ((eLineX & minor) == minor)
        for (double dfX = dfXMinorMin; dfX <= dfXMinorMax; dfX += dfMinorX)
            {
            oPathMinor.moveTo(dfX * dfXScale, (dfYZero - dfYMinorMin) * dfYScale);
            oPathMinor.lineTo(dfX * dfXScale, (dfYZero - dfYMinorMax) * dfYScale);
            }
    if ((eLineY & minor) == minor)
        for (double dfY = dfYMinorMin; dfY <= dfYMinorMax; dfY += dfMinorY)
            {
            oPathMinor.moveTo(dfXMinorMin * dfXScale, (dfYZero - dfY) * dfYScale);
            oPathMinor.lineTo(dfXMinorMax * dfXScale, (dfYZero - dfY) * dfYScale);
            }
    oPathMinor.moveTo(0., 0.);
    QGraphicsPathItem *pItem = pScene->addPath(oPathMinor, oMinorPen);
    pItem->setZValue(dfMinorZ);

    QPainterPath oPathMajor;
    double dfXMajorMin = ceil(m_dfXMin / dfMajorX) * dfMajorX;
    double dfXMajorMax = floor(m_dfXMax / dfMajorX) * dfMajorX;
    double dfYMajorMin = ceil(m_dfYMin / dfMajorY) * dfMajorY;
    double dfYMajorMax = floor(m_dfYMax / dfMajorY) * dfMajorY;
    if ((eLineX & major) == major)
        for (double dfX = dfXMajorMin; dfX <= dfXMajorMax; dfX += dfMajorX)
            {
            oPathMajor.moveTo(dfX * dfXScale, (dfYZero - dfYMinorMin) * dfYScale);
            oPathMajor.lineTo(dfX * dfXScale, (dfYZero - dfYMinorMax) * dfYScale);
            }
    if ((eLineY & major) == major)
        for (double dfY = dfYMajorMin; dfY <= dfYMajorMax; dfY += dfMajorY)
            {
            oPathMajor.moveTo(dfXMinorMin * dfXScale, (dfYZero - dfY) * dfYScale);
            oPathMajor.lineTo(dfXMinorMax * dfXScale, (dfYZero - dfY) * dfYScale);
            }
    oPathMajor.moveTo(0., 0.);
    pItem = pScene->addPath(oPathMajor, oMajorPen);
    pItem->setZValue(dfMajorZ);

    if ((eNumX & min) == min)
        {
        for (double dfX = dfXMajorMin; dfX <= dfXMajorMax; dfX += dfMajorX)
            {
            QGraphicsTextItem *pItem = pScene->addText(QString::number(dfX));
            pItem->setDefaultTextColor(oNumberColor);
            pItem->setPos(dfX * dfXScale - pItem->boundingRect().width() / 2.,
                (dfYZero - dfYMinorMin) * dfYScale + 2.);
            pItem->setZValue(dfMajorZ);
            }
        }
    if ((eNumX & max) == max)
        {
        for (double dfX = dfXMajorMin; dfX <= dfXMajorMax; dfX += dfMajorX)
            {
            QGraphicsTextItem *pItem = pScene->addText(QString::number(dfX));
            pItem->setDefaultTextColor(oNumberColor);
            pItem->setPos(dfX * dfXScale - pItem->boundingRect().width() / 2.,
                (dfYZero - dfYMinorMax) * dfYScale - 1.);
            pItem->setZValue(dfMajorZ);
            }
        }
    if ((eNumY & min) == min)
        {
        for (double dfY = dfYMajorMin; dfY <= dfYMajorMax; dfY += dfMajorY)
            {
            QGraphicsTextItem *pItem = pScene->addText(QString::number(dfY));
            pItem->setDefaultTextColor(oNumberColor);
            pItem->setPos(dfXScale * dfXMinorMin - 2. - pItem->boundingRect().width(),
                (dfYZero - dfY) * dfYScale - pItem->boundingRect().height() / 2.);
            pItem->setZValue(dfMajorZ);
            }
        }
    if ((eNumY & max) == max)
        {
        for (double dfY = dfYMajorMin; dfY <= dfYMajorMax; dfY += dfMajorY)
            {
            QGraphicsTextItem *pItem = pScene->addText(QString::number(dfY));
            pItem->setDefaultTextColor(oNumberColor);
            pItem->setPos(dfXScale * dfXMinorMax + 2.,
                (dfYZero - dfY) * dfYScale - pItem->boundingRect().height() / 2.);
            pItem->setZValue(dfMajorZ);
            }
        }
    }




/********************************************************************/
/*!
    \class Chart
    \brief Chart area for visualizing PE and PEC curves

    \sa Graph, Grid
*/

Chart::Chart(QWidget *pParent)
:   QGraphicsView(pParent)
    {
    m_pRenderTimer = new QTimer(this);
    m_pRenderTimer->setSingleShot(true);
    m_bNeedSceneRender = false;
    connect(m_pRenderTimer, SIGNAL(timeout()), this, SLOT(performSceneRender()));
    }


void Chart::clear()
    {
    QGraphicsScene *pScene = scene();
    if (pScene == 0)
        return;

    QList<QGraphicsItem *> oItemList = pScene->items();
    for (int nItem = 0; nItem < oItemList.size(); nItem++)
        {
        QGraphicsItem *pItem = oItemList.at(nItem);
        pScene->removeItem(pItem);
        delete pItem;
        }
    }


void Chart::resizeEvent(QResizeEvent *pEvent)
    {
    QGraphicsView::resizeEvent(pEvent);
    requestSceneRender();
    }


void Chart::requestSceneRender()
    {
    // do some simple load balancing:
    // if last scene render is younger than 50 msecs,
    // delay next scene render by at least 50 msecs
    m_bNeedSceneRender = true;
    if (m_pRenderTimer->isActive())
        m_pRenderTimer->stop();
    else
        performSceneRender();
    m_pRenderTimer->start(50);
    }


void Chart::performSceneRender()
    {
    // need to check and reset m_bNeedSceneRender
    // to avoid double refreshes when m_pRenderTimer timed out
    // without being preempted by a new render request
    bool bNeedSceneRender = m_bNeedSceneRender;
    m_bNeedSceneRender = false;
    if ((bNeedSceneRender) && (scene() != 0))
        {
        scene()->setSceneRect(scene()->itemsBoundingRect());
        fitInView(sceneRect());
        }
    }


